<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class ExportRequirements extends Command
{
    protected $signature = 'app:export-requirements';
    protected $description = 'Export project requirements for future developers';

    public function handle()
    {
        $requirements = [
            'project' => [
                'name' => 'Restaurant Management System',
                'version' => '1.0.0',
                'laravel_version' => app()->version(),
                'php_required' => '^8.1',
                'database' => 'MySQL 8.0+ / MariaDB 10.4+',
            ],
            'server_requirements' => [
                'php_extensions' => [
                    'bcmath', 'ctype', 'curl', 'dom', 'fileinfo',
                    'json', 'mbstring', 'openssl', 'pdo', 'tokenizer', 'xml'
                ],
                'php_version' => '8.1+',
                'database' => 'MySQL 8.0+ or MariaDB 10.4+',
                'web_server' => 'Apache/Nginx',
                'composer' => '2.0+',
            ],
            'installation_steps' => [
                '1. Clone repository',
                '2. Run: composer install',
                '3. Copy .env.example to .env',
                '4. Set database credentials in .env',
                '5. Run: php artisan key:generate',
                '6. Run: php artisan migrate --seed',
                '7. Run: php artisan storage:link',
            ],
            'migration_commands' => [
                'fresh_install' => 'php artisan migrate:fresh --seed',
                'standard_migration' => 'php artisan migrate',
                'rollback' => 'php artisan migrate:rollback',
                'seed_only' => 'php artisan db:seed',
                'import_sql_dump' => 'php artisan db:seed --class=ImportSqlDumpSeeder',
            ],
            'default_users' => [
                'admin' => 'Full system administrator',
                'denis' => 'Cashier - sales verification',
                'beatrice' => 'Waiter - order management',
                'asha' => 'Waiter - order management',
            ],
            'important_notes' => [
                'Change default passwords after first login',
                'Set APP_ENV=production for live deployment',
                'Configure QUEUE_CONNECTION for background jobs',
                'Set proper file permissions for storage',
                'Regularly backup database',
            ]
        ];

        $json = json_encode($requirements, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        
        File::put(base_path('project_requirements.json'), $json);
        
        // Also create a text version
        $text = $this->generateTextVersion($requirements);
        File::put(base_path('PROJECT_REQUIREMENTS.md'), $text);
        
        $this->info('Requirements files generated successfully!');
        $this->line(' - project_requirements.json');
        $this->line(' - PROJECT_REQUIREMENTS.md');
    }
    
    protected function generateTextVersion($requirements): string
    {
        $text = "# Project Requirements\n\n";
        
        foreach ($requirements as $section => $content) {
            $text .= "## " . ucfirst(str_replace('_', ' ', $section)) . "\n\n";
            
            if (is_array($content)) {
                foreach ($content as $key => $value) {
                    if (is_array($value)) {
                        $text .= "### " . ucfirst(str_replace('_', ' ', $key)) . "\n";
                        foreach ($value as $item) {
                            $text .= "- $item\n";
                        }
                        $text .= "\n";
                    } else {
                        $text .= "**" . ucfirst(str_replace('_', ' ', $key)) . "**: $value\n\n";
                    }
                }
            } else {
                $text .= "$content\n\n";
            }
        }
        
        return $text;
    }
}