<?php

namespace App\Exports;

use App\Models\Stocktaking;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

class StocktakingExport implements FromCollection, WithHeadings, WithMapping
{
    protected $type;
    protected $date;

    public function __construct($type, $date)
    {
        $this->type = $type;
        $this->date = $date;
    }

    public function collection()
    {
        $query = Stocktaking::with('product')
            ->select('stocktakings.*')
            ->join('products', 'stocktakings.product_id', '=', 'products.id')
            ->orderBy('products.product_name');

        switch ($this->type) {
            case 'day':
                $query->whereDate('stocktakings.date', $this->date);
                break;
            case 'week':
                $weekDates = $this->getWeekDates($this->date);
                $query->whereBetween('stocktakings.date', [$weekDates['start'], $weekDates['end']]);
                break;
            case 'month':
                $month = Carbon::parse($this->date)->format('Y-m');
                $query->whereRaw("DATE_FORMAT(stocktakings.date, '%Y-%m') = ?", [$month]);
                break;
        }

        return $query->get();
    }

    public function headings(): array
    {
        return [
            'Product Name',
            'Date',
            'Opening Stock',
            'Additional Stock',
            'Buying Price',
            'Selling Price',
            'Sold Quantity',
            'Amount Obtained',
            'Profit',
            'Closed Stock'
        ];
    }

    public function map($stocktaking): array
    {
        $amountObtained = $stocktaking->sold_quantity * $stocktaking->selling_price;
        $profit = $amountObtained - ($stocktaking->sold_quantity * $stocktaking->buying_price);
        $closedStock = ($stocktaking->opening_stock + $stocktaking->additional_stock) - $stocktaking->sold_quantity;

        return [
            $stocktaking->product->product_name,
            $stocktaking->date->format('Y-m-d'),
            $stocktaking->opening_stock,
            $stocktaking->additional_stock,
            number_format($stocktaking->buying_price, 2),
            number_format($stocktaking->selling_price, 2),
            $stocktaking->sold_quantity,
            number_format($amountObtained, 2),
            number_format($profit, 2),
            $closedStock
        ];
    }

    private function getWeekDates($date)
    {
        $carbonDate = Carbon::parse($date);
        return [
            'start' => $carbonDate->startOfWeek()->format('Y-m-d'),
            'end' => $carbonDate->endOfWeek()->format('Y-m-d')
        ];
    }
}