<?php

namespace App\Http\Controllers;

use App\Models\Stocktaking;
use App\Models\Product;
use Illuminate\Http\Request;
use Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class CashierStockController extends Controller
{
    public function index()
    {
        return view('stocktaking.cashier-entry');
    }

    public function getTodayStockData()
    {
        try {
            $today = Carbon::today()->format('Y-m-d');
            
            // Get breakfast products only
            $products = Product::join('categories', 'products.category_id', '=', 'categories.id')
                ->where('categories.category_name', 'Breakfast')
                ->select('products.*')
                ->get();

            $stocktakingData = [];

            foreach ($products as $product) {
                $stockRecord = Stocktaking::where('product_id', $product->id)
                    ->whereDate('date', $today)
                    ->first();

                if (!$stockRecord) {
                    $stockRecord = Stocktaking::create([
                        'product_id' => $product->id,
                        'date' => $today,
                        'opening_stock' => 0,
                        'additional_stock' => 0,
                        'buying_price' => $product->purchase_price,
                        'selling_price' => $product->selling_price,
                        'sold_quantity' => 0,
                        'created_by' => Auth::id()
                    ]);
                }

                // Check if opening stock was already filled today
                $openingStockFilled = $stockRecord->opening_stock > 0;
                
                // Get sold quantity from sales
                $soldQuantity = DB::table('sale_items')
                    ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
                    ->where('sale_items.product_id', $product->id)
                    ->whereDate('sales.created_at', $today)
                    ->where('sales.status', 'verified')
                    ->sum('sale_items.quantity');

                $stocktakingData[] = [
                    'id' => $stockRecord->id,
                    'product_id' => $product->id,
                    'product_name' => $product->product_name,
                    'opening_stock' => (int)$stockRecord->opening_stock,
                    'additional_stock' => (int)$stockRecord->additional_stock,
                    'buying_price' => (float)$stockRecord->buying_price,
                    'selling_price' => (float)$stockRecord->selling_price,
                    'sold_quantity' => (int)$soldQuantity,
                    'amount_obtained' => (float)($soldQuantity * $stockRecord->selling_price),
                    'profit' => (float)(($soldQuantity * $stockRecord->selling_price) - ($soldQuantity * $stockRecord->buying_price)),
                    'closed_stock' => (int)(($stockRecord->opening_stock + $stockRecord->additional_stock) - $soldQuantity),
                    'opening_stock_filled' => $openingStockFilled,
                    'can_edit_opening' => !$openingStockFilled
                ];
            }

            return response()->json($stocktakingData);

        } catch (\Exception $e) {
            Log::error("Error in getTodayStockData: " . $e->getMessage());
            return response()->json(['error' => 'Error loading stock data'], 500);
        }
    }

    public function updateStock(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'opening_stock' => 'nullable|integer|min:0',
            'additional_stock' => 'nullable|integer|min:0'
        ]);

        DB::beginTransaction();

        try {
            $today = Carbon::today()->format('Y-m-d');
            
            $stockRecord = Stocktaking::where('product_id', $request->product_id)
                ->whereDate('date', $today)
                ->firstOrFail();

            // Check if opening stock was already filled
            if ($request->has('opening_stock') && $stockRecord->opening_stock > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Opening stock cannot be modified once filled for the day'
                ], 400);
            }

            $updateData = [];
            if ($request->has('opening_stock')) {
                $updateData['opening_stock'] = $request->opening_stock;
            }
            if ($request->has('additional_stock')) {
                $updateData['additional_stock'] = $request->additional_stock;
            }

            $stockRecord->update($updateData);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Stock updated successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Error in updateStock: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating stock'
            ], 500);
        }
    }

    public function resetForNewDay()
    {
        try {
            $today = Carbon::today()->format('Y-m-d');
            
            // This would typically be called automatically via a scheduled job
            // For now, we'll just return today's data
            return response()->json([
                'success' => true,
                'message' => 'System ready for new day'
            ]);

        } catch (\Exception $e) {
            Log::error("Error in resetForNewDay: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error resetting for new day'
            ], 500);
        }
    }
}