<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sale;
use App\Models\User;
use App\Models\Category;
use App\Services\SalesChartService;
use Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    protected $salesChartService;
    
    public function __construct(SalesChartService $salesChartService)
    {
        $this->salesChartService = $salesChartService;
    }

   public function dashboard(Request $request)
{
    $user = Auth::user();

    // Allow only Admin, Cashier, Waiter
    if (!in_array($user->is_role, [1, 2, 3])) {
        abort(403, 'Unauthorized access');
    }

    try {
        // Recent sales
        $sales = Sale::latest()->cursorPaginate(500);

        // Chart data
        $chartData = [
            'today_vs_yesterday' => $this->getTodayYesterdayData(),
            'this_week_vs_last_week' => $this->getThisWeekLastWeekData(),
            'this_month_vs_last_month' => $this->getThisMonthLastMonthData(),
            'this_year_vs_last_year' => $this->getThisYearLastYearData(),
        ];

        // Sales status (for gauge)
        $salesStatus = $this->getSalesStatus();

        // Summary statistics
        $stats = $this->calculateDashboardStats();

        // Waiter performance
        $waiterPerformance = $this->getWaiterPerformance();

        // Determine view by role
        $view = $this->getDashboardView($user->is_role);

        // 🔹 Add this block (available for cashier & admin)
        $waiterStats = Sale::select('waiter_id',
                DB::raw('COUNT(*) as pending_count'),
                DB::raw('SUM(total) as pending_total')
            )
            ->where('status', 'pending')
            ->groupBy('waiter_id')
            ->with('waiter:id,name')
            ->get();

        // 🔹 If waiter, add personal stats
        if ($user->is_role == 3) {
            $waiterData = $this->getWaiterDashboardData($user->id);

            return view($view, array_merge(
                [
                    'user' => $user,
                    'sales' => $sales,
                    'chartData' => $chartData,
                    'stats' => $stats,
                    'salesStatus' => $salesStatus,
                    'waiterPerformance' => $waiterPerformance,
                    'waiterStats' => $waiterStats,
                ],
                $waiterData
            ));
        }

        // 🔹 Admin or cashier
        return view($view, compact(
            'user',
            'sales',
            'chartData',
            'stats',
            'salesStatus',
            'waiterPerformance',
            'waiterStats'
        ));

    } catch (\Exception $e) {
        \Log::error('Dashboard error: ' . $e->getMessage());

        $view = $this->getDashboardView(Auth::user()->is_role);

        return view($view, [
            'user' => Auth::user(),
            'sales' => collect([]),
            'chartData' => $this->getFallbackChartData(),
            'stats' => $this->getFallbackStats(),
            'salesStatus' => ['verified' => 0, 'pending' => 0, 'total' => 0],
            'waiterPerformance' => [],
            'waiterStats' => [],
        ]);
    }
}

public function verify(Sale $sale)
{
    // Check if user is authorized (admin or cashier only)
    $user = Auth::user();
    if (!in_array($user->is_role, [1, 2])) { // 1=Admin, 2=Cashier
        return response()->json([
            'success' => false, 
            'message' => 'Unauthorized: Only admin and cashier can verify sales.'
        ], 403);
    }

    try {
        if ($sale->status === 'pending') {
            $sale->update([
                'cashier_id' => auth()->id(),
                'status' => 'verified',
                'verified_at' => Carbon::now()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Sale verified successfully'
            ]);
        }
        
        return response()->json([
            'success' => false, 
            'message' => 'Sale is already verified.'
        ], 400);
        
    } catch (\Exception $e) {
        \Log::error('Sale verification error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false,
            'message' => 'Server error occurred while verifying sale.'
        ], 500);
    }
}

    // API endpoints for AJAX updates
    public function getChartData(Request $request)
    {
        $type = $request->get('type', 'today_vs_yesterday');
        
        try {
            switch ($type) {
                case 'today_vs_yesterday':
                    $data = $this->getTodayYesterdayData();
                    break;
                case 'this_week_vs_last_week':
                    $data = $this->getThisWeekLastWeekData();
                    break;
                case 'this_month_vs_last_month':
                    $data = $this->getThisMonthLastMonthData();
                    break;
                case 'this_year_vs_last_year':
                    $data = $this->getThisYearLastYearData();
                    break;
                default:
                    $data = $this->getTodayYesterdayData();
            }

            return response()->json([
                'success' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch chart data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // ==================== PRIVATE METHODS ====================

    /**
     * Determine which dashboard view to show based on user role
     */
    private function getDashboardView($role)
    {
        switch ($role) {
            case 1: // Admin
                return 'dashboard.admin_list';
            case 2: // Cashier
                return 'dashboard.cashier_list';
            case 3: // Waiter
                return 'dashboard.waiter_list';
            default:
                return 'dashboard.admin_list';
        }
    }


    // ==================== GENERAL DASHBOARD METHODS ====================

    /**
     * Today vs Yesterday (7:00 - 22:00 hourly)
     */
    private function getTodayYesterdayData()
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();
        
        // Get today's sales by hour (7:00 - 22:00)
        $todaySales = Sale::whereDate('created_at', $today)
            ->select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('SUM(total) as total')
            )
            ->whereBetween(DB::raw('HOUR(created_at)'), [7, 22])
            ->groupBy('hour')
            ->orderBy('hour')
            ->pluck('total', 'hour')
            ->toArray();

        // Get yesterday's sales by hour (7:00 - 22:00)
        $yesterdaySales = Sale::whereDate('created_at', $yesterday)
            ->select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('SUM(total) as total')
            )
            ->whereBetween(DB::raw('HOUR(created_at)'), [7, 22])
            ->groupBy('hour')
            ->orderBy('hour')
            ->pluck('total', 'hour')
            ->toArray();

        // Fill missing hours with 0
        $todayData = [];
        $yesterdayData = [];
        $labels = [];
        
        for ($hour = 7; $hour <= 22; $hour++) {
            $labels[] = sprintf('%02d:00', $hour);
            $todayData[] = $todaySales[$hour] ?? 0;
            $yesterdayData[] = $yesterdaySales[$hour] ?? 0;
        }

        return [
            'labels' => $labels,
            'today_sales' => $todayData,
            'yesterday_sales' => $yesterdayData,
            'today_date' => $today->format('Y-m-d'),
            'yesterday_date' => $yesterday->format('Y-m-d'),
        ];
    }

    /**
     * This Week vs Last Week (7 days)
     */
    private function getThisWeekLastWeekData()
    {
        $currentWeekStart = Carbon::now()->startOfWeek();
        $currentWeekEnd = Carbon::now()->endOfWeek();
        $lastWeekStart = Carbon::now()->subWeek()->startOfWeek();
        $lastWeekEnd = Carbon::now()->subWeek()->endOfWeek();

        // Get current week sales
        $currentWeekSales = Sale::whereBetween('created_at', [$currentWeekStart, $currentWeekEnd])
            ->select(
                DB::raw('DAYNAME(created_at) as day'),
                DB::raw('DAYOFWEEK(created_at) as day_num'),
                DB::raw('SUM(total) as total')
            )
            ->groupBy('day', 'day_num')
            ->orderBy('day_num')
            ->pluck('total', 'day')
            ->toArray();

        // Get last week sales
        $lastWeekSales = Sale::whereBetween('created_at', [$lastWeekStart, $lastWeekEnd])
            ->select(
                DB::raw('DAYNAME(created_at) as day'),
                DB::raw('DAYOFWEEK(created_at) as day_num'),
                DB::raw('SUM(total) as total')
            )
            ->groupBy('day', 'day_num')
            ->orderBy('day_num')
            ->pluck('total', 'day')
            ->toArray();

        $days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        
        $currentWeekData = [];
        $lastWeekData = [];
        
        foreach ($days as $day) {
            $currentWeekData[] = $currentWeekSales[$day] ?? 0;
            $lastWeekData[] = $lastWeekSales[$day] ?? 0;
        }

        return [
            'labels' => array_map(fn($day) => substr($day, 0, 3), $days),
            'this_week_sales' => $currentWeekData,
            'last_week_sales' => $lastWeekData,
            'this_week_range' => $currentWeekStart->format('M d') . ' - ' . $currentWeekEnd->format('M d'),
            'last_week_range' => $lastWeekStart->format('M d') . ' - ' . $lastWeekEnd->format('M d'),
        ];
    }

    /**
     * This Month vs Last Month (4 weeks)
     */
    private function getThisMonthLastMonthData()
    {
        $currentMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();
        
        // Get current month sales by week
        $currentMonthSales = Sale::whereYear('created_at', $currentMonth->year)
            ->whereMonth('created_at', $currentMonth->month)
            ->select(
                DB::raw('WEEK(created_at, 1) - WEEK(DATE_SUB(created_at, INTERVAL DAYOFMONTH(created_at)-1 DAY), 1) + 1 as week'),
                DB::raw('SUM(total) as total')
            )
            ->groupBy('week')
            ->orderBy('week')
            ->pluck('total', 'week')
            ->toArray();

        // Get last month sales by week
        $lastMonthSales = Sale::whereYear('created_at', $lastMonth->year)
            ->whereMonth('created_at', $lastMonth->month)
            ->select(
                DB::raw('WEEK(created_at, 1) - WEEK(DATE_SUB(created_at, INTERVAL DAYOFMONTH(created_at)-1 DAY), 1) + 1 as week'),
                DB::raw('SUM(total) as total')
            )
            ->groupBy('week')
            ->orderBy('week')
            ->pluck('total', 'week')
            ->toArray();

        $weeks = ['Week 1', 'Week 2', 'Week 3', 'Week 4'];
        
        $currentMonthData = [];
        $lastMonthData = [];
        
        for ($week = 1; $week <= 4; $week++) {
            $currentMonthData[] = $currentMonthSales[$week] ?? 0;
            $lastMonthData[] = $lastMonthSales[$week] ?? 0;
        }

        return [
            'labels' => $weeks,
            'this_month_sales' => $currentMonthData,
            'last_month_sales' => $lastMonthData,
            'this_month' => $currentMonth->format('F Y'),
            'last_month' => $lastMonth->format('F Y'),
        ];
    }

    /**
     * This Year vs Last Year (12 months)
     */
    private function getThisYearLastYearData()
    {
        $currentYear = Carbon::now()->year;
        $lastYear = Carbon::now()->subYear()->year;

        // Get current year sales by month
        $currentYearSales = Sale::whereYear('created_at', $currentYear)
            ->select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('SUM(total) as total')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month')
            ->toArray();

        // Get last year sales by month
        $lastYearSales = Sale::whereYear('created_at', $lastYear)
            ->select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('SUM(total) as total')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month')
            ->toArray();

        $months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
        
        $currentYearData = [];
        $lastYearData = [];
        
        for ($month = 1; $month <= 12; $month++) {
            $currentYearData[] = $currentYearSales[$month] ?? 0;
            $lastYearData[] = $lastYearSales[$month] ?? 0;
        }

        return [
            'labels' => $months,
            'this_year_sales' => $currentYearData,
            'last_year_sales' => $lastYearData,
            'this_year' => $currentYear,
            'last_year' => $lastYear,
        ];
    }

    /**
     * Calculate comprehensive dashboard statistics
     */
    private function calculateDashboardStats()
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();
        
        // Get today's total sales
        $todaySales = Sale::whereDate('created_at', $today)->sum('total');
        $yesterdaySales = Sale::whereDate('created_at', $yesterday)->sum('total');
        
        // Calculate today's sold items (sum of all quantities from sale items)
        $todayItemsSold = Sale::whereDate('created_at', $today)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
        
        // Calculate yesterday's sold items for growth comparison
        $yesterdayItemsSold = Sale::whereDate('created_at', $yesterday)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
        
        // Get today's number of sales (transactions)
        $todaySalesCount = Sale::whereDate('created_at', $today)->count();
        $yesterdaySalesCount = Sale::whereDate('created_at', $yesterday)->count();
        
        // Calculate growth percentages
        $salesGrowth = $this->calculateGrowth($yesterdaySales, $todaySales);
        $itemsGrowth = $this->calculateGrowth($yesterdayItemsSold, $todayItemsSold);
        $salesCountGrowth = $this->calculateGrowth($yesterdaySalesCount, $todaySalesCount);

 // BREAKFAST SALES
        $breakfastCategoryId = Category::whereRaw('LOWER(category_name) = ?', ['breakfast'])->value('id');

        $todayBreakfastSales = Sale::whereDate('created_at', $today)
            ->whereHas('items.product.category', function ($q) use ($breakfastCategoryId) {
                $q->where('id', $breakfastCategoryId);
            })
            ->with(['items' => function ($q) use ($breakfastCategoryId) {
                $q->whereHas('product.category', function ($q2) use ($breakfastCategoryId) {
                    $q2->where('id', $breakfastCategoryId);
                });
            }])
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum(function($item) {
                    return $item->quantity * $item->price;
                });
            });

        $yesterdayBreakfastSales = Sale::whereDate('created_at', $yesterday)
            ->whereHas('items.product.category', function ($q) use ($breakfastCategoryId) {
                $q->where('id', $breakfastCategoryId);
            })
            ->with(['items' => function ($q) use ($breakfastCategoryId) {
                $q->whereHas('product.category', function ($q2) use ($breakfastCategoryId) {
                    $q2->where('id', $breakfastCategoryId);
                });
            }])
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum(function($item) {
                    return $item->quantity * $item->price;
                });
            });

    $breakfastGrowth = $this->calculateGrowth($yesterdayBreakfastSales, $todayBreakfastSales);


        return [
            'today_sales' => $todaySales,
            'today_items_sold' => $todayItemsSold,
            'breakfast_sales' => $todayBreakfastSales,
            'breakfast_sales_yesterday' => $yesterdayBreakfastSales,
            'breakfast_growth' => $breakfastGrowth,
            'today_sales_count' => $todaySalesCount,
            'yesterday_sales' => $yesterdaySales,
            'yesterday_items_sold' => $yesterdayItemsSold,
            'yesterday_sales_count' => $yesterdaySalesCount,
            'sales_growth' => $salesGrowth,
            'items_growth' => $itemsGrowth,
            'sales_count_growth' => $salesCountGrowth,
            'average_sale_value' => $todaySalesCount > 0 ? $todaySales / $todaySalesCount : 0,
        ];
    }

    /**
     * Get sales status for gauge chart
     */
private function getSalesStatus()
{
    // Get only today's sales
    $sales = Sale::select('status')
        ->whereDate('created_at', Carbon::today())
        ->get();

    $total = $sales->count();

    if ($total === 0) {
        return [
            'verified' => 0,
            'pending' => 0,
            'total' => 0,
            'verified_count' => 0,
            'pending_count' => 0
        ];
    }

    $verified = $sales->where('status', 'verified')->count();
    $pending = $sales->where('status', 'pending')->count();

    return [
        'verified' => round(($verified / $total) * 100),
        'pending' => round(($pending / $total) * 100),
        'total' => $total,
        'verified_count' => $verified,
        'pending_count' => $pending
    ];
}

    /**
     * Get waiter performance for today - ONLY ACTIVE WAITERS (status = 1)
     */
    private function getWaiterPerformance()
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();
        
        // Get all ACTIVE waiters (users with is_role 2 or 3 AND status = 1)
        $waiters = User::whereIn('is_role', [2, 3])
            ->where('status', 1) // Only active users
            ->get();
        
        $waiterPerformance = [];
        
        foreach ($waiters as $waiter) {
            // Today's performance
            $todaySales = Sale::whereDate('created_at', $today)
                ->where('waiter_id', $waiter->id)
                ->with('items')
                ->get();
            
            $todayTotalAmount = $todaySales->sum('total');
            $todaySalesCount = $todaySales->count();
            $todayItemsSold = $todaySales->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
            
            // Yesterday's performance
            $yesterdaySales = Sale::whereDate('created_at', $yesterday)
                ->where('waiter_id', $waiter->id)
                ->with('items')
                ->get();
            
            $yesterdayTotalAmount = $yesterdaySales->sum('total');
            $yesterdaySalesCount = $yesterdaySales->count();
            $yesterdayItemsSold = $yesterdaySales->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
            
            // Calculate growth percentages
            $salesGrowth = $this->calculateGrowth($yesterdaySalesCount, $todaySalesCount);
            $amountGrowth = $this->calculateGrowth($yesterdayTotalAmount, $todayTotalAmount);
            
            $waiterPerformance[] = [
                'waiter_name' => $waiter->name,
                'today_total_amount' => $todayTotalAmount,
                'today_sales_count' => $todaySalesCount,
                'today_items_sold' => $todayItemsSold,
                'yesterday_total_amount' => $yesterdayTotalAmount,
                'yesterday_sales_count' => $yesterdaySalesCount,
                'yesterday_items_sold' => $yesterdayItemsSold,
                'sales_growth' => $salesGrowth,
                'amount_growth' => $amountGrowth,
                'average_sale_value' => $todaySalesCount > 0 ? $todayTotalAmount / $todaySalesCount : 0,
            ];
        }
        
        // Sort by today's total amount (highest first)
        usort($waiterPerformance, function($a, $b) {
            return $b['today_total_amount'] <=> $a['today_total_amount'];
        });
        
        return $waiterPerformance;
    }

    // ==================== WAITER-SPECIFIC METHODS ====================

    /**
     * Get waiter-specific dashboard data
     */
    private function getWaiterDashboardData($userId)
    {
        return [
            'waiterStats' => $this->getWaiterStats($userId),
            'waiterChartData' => $this->getWaiterChartData($userId),
            'todaySales' => $this->getTodaySales($userId),
            'weeklyPerformance' => $this->getWeeklyPerformance($userId),
            'monthlyStats' => $this->getMonthlyStats($userId),
            'monthlyWeeklyProgress' => $this->getMonthlyWeeklyProgress($userId),
            'bestDay' => $this->getBestDay($userId),
        ];
    }

    /**
     * Get waiter-specific statistics
     */
    private function getWaiterStats($userId)
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();
        
        // Today's data
        $todaySalesAmount = Sale::whereDate('created_at', $today)
            ->where('waiter_id', $userId)
            ->sum('total');
        
        $todayItemsSold = Sale::whereDate('created_at', $today)
            ->where('waiter_id', $userId)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
        
        $todayTransactions = Sale::whereDate('created_at', $today)
            ->where('waiter_id', $userId)
            ->count();
        
        // Yesterday's data
        $yesterdaySalesAmount = Sale::whereDate('created_at', $yesterday)
            ->where('waiter_id', $userId)
            ->sum('total');
        
        $yesterdayItemsSold = Sale::whereDate('created_at', $yesterday)
            ->where('waiter_id', $userId)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
        
        $yesterdayTransactions = Sale::whereDate('created_at', $yesterday)
            ->where('waiter_id', $userId)
            ->count();
        
        // Calculate growth
        $salesGrowth = $this->calculateGrowth($yesterdaySalesAmount, $todaySalesAmount);
        $itemsGrowth = $this->calculateGrowth($yesterdayItemsSold, $todayItemsSold);
        $transactionsGrowth = $this->calculateGrowth($yesterdayTransactions, $todayTransactions);
        
        return [
            'today_sales' => $todaySalesAmount,
            'today_items_sold' => $todayItemsSold,
            'today_transactions' => $todayTransactions,
            'yesterday_sales' => $yesterdaySalesAmount,
            'yesterday_items_sold' => $yesterdayItemsSold,
            'yesterday_transactions' => $yesterdayTransactions,
            'sales_growth' => $salesGrowth,
            'items_growth' => $itemsGrowth,
            'transactions_growth' => $transactionsGrowth,
            'average_sale_value' => $todayTransactions > 0 ? $todaySalesAmount / $todayTransactions : 0,
        ];
    }

    /**
     * Get waiter-specific chart data
     */
    private function getWaiterChartData($userId)
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();
        
        // Get today's sales by hour
        $todaySales = Sale::whereDate('created_at', $today)
            ->where('waiter_id', $userId)
            ->select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('SUM(total) as total')
            )
            ->whereBetween(DB::raw('HOUR(created_at)'), [7, 22])
            ->groupBy('hour')
            ->orderBy('hour')
            ->pluck('total', 'hour')
            ->toArray();

        // Get yesterday's sales by hour
        $yesterdaySales = Sale::whereDate('created_at', $yesterday)
            ->where('waiter_id', $userId)
            ->select(
                DB::raw('HOUR(created_at) as hour'),
                DB::raw('SUM(total) as total')
            )
            ->whereBetween(DB::raw('HOUR(created_at)'), [7, 22])
            ->groupBy('hour')
            ->orderBy('hour')
            ->pluck('total', 'hour')
            ->toArray();

        // Fill missing hours
        $todayData = [];
        $yesterdayData = [];
        $labels = [];
        
        for ($hour = 7; $hour <= 22; $hour++) {
            $labels[] = sprintf('%02d:00', $hour);
            $todayData[] = $todaySales[$hour] ?? 0;
            $yesterdayData[] = $yesterdaySales[$hour] ?? 0;
        }

        return [
            'today_vs_yesterday' => [
                'labels' => $labels,
                'today_sales' => $todayData,
                'yesterday_sales' => $yesterdayData,
            ]
        ];
    }

    /**
     * Get today's Sales for the waiter
     */
    private function getTodaySales($userId)
    {
        return Sale::whereDate('created_at', Carbon::today())
            ->where('waiter_id', $userId)
            ->with(['items.product'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

    /**
     * Get weekly performance data
     */
    private function getWeeklyPerformance($userId)
    {
        $weekStart = Carbon::now()->startOfWeek();
        $weekEnd = Carbon::now()->endOfWeek();
        
        $weeklyData = Sale::whereBetween('created_at', [$weekStart, $weekEnd])
            ->where('waiter_id', $userId)
            ->select(
                DB::raw('DAYNAME(created_at) as day'),
                DB::raw('DAYOFWEEK(created_at) as day_num'),
                DB::raw('SUM(total) as total_sales'),
                DB::raw('COUNT(*) as transactions_count')
            )
            ->groupBy('day', 'day_num')
            ->orderBy('day_num')
            ->get()
            ->map(function($item) {
                return [
                    'day' => substr($item->day, 0, 3),
                    'sales' => $item->total_sales,
                    'transactions' => $item->transactions_count
                ];
            });
        
        // Ensure all days are present
        $days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
        $fullWeek = collect($days)->map(function($day) use ($weeklyData) {
            $existing = $weeklyData->firstWhere('day', $day);
            return $existing ?: ['day' => $day, 'sales' => 0, 'transactions' => 0];
        });
        
        return $fullWeek;
    }

    /**
     * Get monthly statistics
     */
    private function getMonthlyStats($userId)
    {
        $currentMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();
        
        // Current month data
        $currentMonthData = Sale::whereYear('created_at', $currentMonth->year)
            ->whereMonth('created_at', $currentMonth->month)
            ->where('waiter_id', $userId)
            ->select(
                DB::raw('SUM(total) as total_sales'),
                DB::raw('COUNT(*) as total_transactions')
            )
            ->first();
        
        $currentMonthItems = Sale::whereYear('created_at', $currentMonth->year)
            ->whereMonth('created_at', $currentMonth->month)
            ->where('waiter_id', $userId)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
        
        // Last month data
        $lastMonthData = Sale::whereYear('created_at', $lastMonth->year)
            ->whereMonth('created_at', $lastMonth->month)
            ->where('waiter_id', $userId)
            ->select(
                DB::raw('SUM(total) as total_sales'),
                DB::raw('COUNT(*) as total_transactions')
            )
            ->first();
        
        $lastMonthItems = Sale::whereYear('created_at', $lastMonth->year)
            ->whereMonth('created_at', $lastMonth->month)
            ->where('waiter_id', $userId)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });
        
        // Calculate growth
        $salesGrowth = $this->calculateGrowth($lastMonthData->total_sales ?? 0, $currentMonthData->total_sales ?? 0);
        $transactionsGrowth = $this->calculateGrowth($lastMonthData->total_transactions ?? 0, $currentMonthData->total_transactions ?? 0);
        $itemsGrowth = $this->calculateGrowth($lastMonthItems, $currentMonthItems);
        
        return [
            'total_sales' => $currentMonthData->total_sales ?? 0,
            'total_transactions' => $currentMonthData->total_transactions ?? 0,
            'total_items' => $currentMonthItems,
            'sales_growth' => $salesGrowth,
            'transactions_growth' => $transactionsGrowth,
            'items_growth' => $itemsGrowth,
            'average_sale' => ($currentMonthData->total_transactions ?? 0) > 0 ? 
                ($currentMonthData->total_sales ?? 0) / ($currentMonthData->total_transactions ?? 0) : 0,
        ];
    }

    /**
     * Get weekly progress for the current month
     */
    private function getMonthlyWeeklyProgress($userId)
    {
        $currentMonth = Carbon::now()->startOfMonth();
        
        $weeklyData = Sale::whereYear('created_at', $currentMonth->year)
            ->whereMonth('created_at', $currentMonth->month)
            ->where('waiter_id', $userId)
            ->select(
                DB::raw('WEEK(created_at, 1) - WEEK(DATE_SUB(created_at, INTERVAL DAYOFMONTH(created_at)-1 DAY), 1) + 1 as week'),
                DB::raw('SUM(total) as total_sales'),
                DB::raw('COUNT(*) as transactions_count')
            )
            ->groupBy('week')
            ->orderBy('week')
            ->get()
            ->map(function($item) {
                return [
                    'week' => 'Week ' . $item->week,
                    'sales' => $item->total_sales,
                    'transactions' => $item->transactions_count
                ];
            });
        
        // Ensure all 4 weeks are present
        $fullMonth = collect();
        for ($week = 1; $week <= 4; $week++) {
            $existing = $weeklyData->firstWhere('week', 'Week ' . $week);
            $fullMonth->push($existing ?: [
                'week' => 'Week ' . $week,
                'sales' => 0,
                'transactions' => 0
            ]);
        }
        
        return $fullMonth;
    }

    /**
     * Get best performing day
     */
    private function getBestDay($userId)
    {
        $bestDay = Sale::where('waiter_id', $userId)
            ->select(
                DB::raw('DAYNAME(created_at) as day'),
                DB::raw('SUM(total) as total_sales')
            )
            ->groupBy('day')
            ->orderBy('total_sales', 'desc')
            ->first();
        
        return $bestDay ? [
            'day' => substr($bestDay->day, 0, 3),
            'sales' => $bestDay->total_sales
        ] : null;
    }

    // ==================== UTILITY METHODS ====================

    /**
     * Get recent sales with cashier information
     */
    private function getRecentSales($limit = 50)
    {
        return Sale::with(['waiter', 'cashier', 'items.product'])
            ->orderBy('created_at', 'desc')
            ->take($limit)
            ->get();
    }

    private function calculateGrowth($previous, $current)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 2);
    }

    private function getFallbackChartData()
    {
        return [
            'today_vs_yesterday' => [
                'labels' => array_map(fn($h) => sprintf('%02d:00', $h), range(7, 22)),
                'today_sales' => array_fill(0, 16, 0),
                'yesterday_sales' => array_fill(0, 16, 0),
            ],
            'this_week_vs_last_week' => [
                'labels' => ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
                'this_week_sales' => array_fill(0, 7, 0),
                'last_week_sales' => array_fill(0, 7, 0),
            ],
            'this_month_vs_last_month' => [
                'labels' => ['Week 1', 'Week 2', 'Week 3', 'Week 4'],
                'this_month_sales' => array_fill(0, 4, 0),
                'last_month_sales' => array_fill(0, 4, 0),
            ],
            'this_year_vs_last_year' => [
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                'this_year_sales' => array_fill(0, 12, 0),
                'last_year_sales' => array_fill(0, 12, 0),
            ]
        ];
    }

    private function getFallbackStats()
    {
        return [
            'today_sales' => 0,
            'today_items_sold' => 0,
            'today_sales_count' => 0,
            'yesterday_sales' => 0,
            'yesterday_items_sold' => 0,
            'yesterday_sales_count' => 0,
            'sales_growth' => 0,
            'items_growth' => 0,
            'sales_count_growth' => 0,
            'average_sale_value' => 0,
        ];
    }
}