<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sale;
use App\Models\User;
use Carbon\Carbon;
use PDF;

class PayrollController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        // Only allow admin access
        if (auth()->user()->is_role != 1) {
            abort(403, 'Unauthorized access.');
        }

        $year = $request->get('year', Carbon::now()->year);
        $week = $request->get('week', Carbon::now()->weekOfYear);

        // Calculate week dates
        $weekData = $this->getWeekDates($year, $week);
        $weekStart = $weekData['start'];
        $weekEnd = $weekData['end'];

        // Get waiters with weekly payroll data
        $waiters = User::whereIn('is_role', [2, 3])
            ->where('status', 1)
            ->get()
            ->map(function ($waiter) use ($weekStart, $weekEnd) {
                return $this->getWaiterPayrollData($waiter, $weekStart, $weekEnd);
            })
            ->filter(function ($waiter) {
                return $waiter['weekly_total_sales'] > 0 || $waiter['name'] === 'aisha juma'; // Include Aisha even if no sales
            })
            ->sortByDesc('weekly_total_sales')
            ->values();

        $availableYears = $this->getAvailableYears();

        return view('admin.payroll.index', compact(
            'waiters', 'year', 'week', 'weekStart', 'weekEnd', 'availableYears'
        ));
    }

    private function getWeekDates($year, $week)
    {
        $startOfWeek = Carbon::now()->setISODate($year, $week)->startOfWeek();
        $endOfWeek = $startOfWeek->copy()->endOfWeek();

        return [
            'start' => $startOfWeek,
            'end' => $endOfWeek
        ];
    }

    private function getWaiterPayrollData($waiter, $weekStart, $weekEnd)
    {
        // Special handling for Aisha Juma and Tumaini George
        if (strtolower($waiter->name) === 'aisha juma') {
            return $this->getAishaJumaPayrollData($waiter, $weekStart, $weekEnd);
        }
        
        if (strtolower($waiter->name) === 'tumaini george') {
            return $this->getTumainiGeorgePayrollData($waiter, $weekStart, $weekEnd);
        }

        // Regular waiters
        $dailyData = $this->getDailyBreakdown($waiter->id, $weekStart, $weekEnd);

        // Weekly totals are SUM of daily totals (NOT re-calculated)
        $weeklyTotalSales = collect($dailyData)->sum('total_amount');
        $weeklyTotalPayment = collect($dailyData)->sum('payment');

        return [
            'id' => $waiter->id,
            'name' => $waiter->name,
            'avatar' => $waiter->avatar,
            'weekly_total_sales' => $weeklyTotalSales,
            'weekly_total_payment' => $weeklyTotalPayment,
            'daily' => $dailyData,
            'payment_type' => 'commission',
            'special_note' => null
        ];
    }

    /**
     * Aisha Juma gets 10,000 TZS fixed payment for each day she has sales
     */
    private function getAishaJumaPayrollData($waiter, $weekStart, $weekEnd)
    {
        $dailyData = [];
        $currentDay = $weekStart->copy();
        $daysWorked = 0;

        while ($currentDay <= $weekEnd) {
            $daySales = Sale::where('waiter_id', $waiter->id)
                ->whereDate('created_at', $currentDay)
                ->get();

            $dayTotal = $daySales->sum('total');
            $hasSales = $daySales->count() > 0;

            // Aisha gets 10,000 TZS for each day she has sales, regardless of amount
            $dayPayment = $hasSales ? 10000 : 0;

            if ($hasSales) {
                $daysWorked++;
                $dailyData[] = [
                    'date' => $currentDay->format('Y-m-d'),
                    'day_name' => $currentDay->format('l'),
                    'total_amount' => $dayTotal,
                    'payment' => $dayPayment,
                    'sales_count' => $daySales->count(),
                    'is_eligible' => true,
                    'note' => 'Fixed daily rate'
                ];
            }

            $currentDay->addDay();
        }

        $weeklyTotalPayment = $daysWorked * 10000;

        return [
            'id' => $waiter->id,
            'name' => $waiter->name,
            'avatar' => $waiter->avatar,
            'weekly_total_sales' => collect($dailyData)->sum('total_amount'),
            'weekly_total_payment' => $weeklyTotalPayment,
            'daily' => $dailyData,
            'payment_type' => 'fixed_daily',
            'special_note' => "Fixed 10,000 TZS per day worked ({$daysWorked} days)",
            'days_worked' => $daysWorked
        ];
    }

    /**
     * Tumaini George is paid fixed amount monthly, so exclude from weekly payroll
     */
    private function getTumainiGeorgePayrollData($waiter, $weekStart, $weekEnd)
    {
        // Check if she had any sales in the week (for reporting purposes)
        $weeklySales = Sale::where('waiter_id', $waiter->id)
            ->whereBetween('created_at', [$weekStart, $weekEnd])
            ->get();

        $dailyData = [];
        $currentDay = $weekStart->copy();

        while ($currentDay <= $weekEnd) {
            $daySales = Sale::where('waiter_id', $waiter->id)
                ->whereDate('created_at', $currentDay)
                ->get();

            $dayTotal = $daySales->sum('total');
            $hasSales = $daySales->count() > 0;

            if ($hasSales) {
                $dailyData[] = [
                    'date' => $currentDay->format('Y-m-d'),
                    'day_name' => $currentDay->format('l'),
                    'total_amount' => $dayTotal,
                    'payment' => 0, // No weekly payment - paid monthly
                    'sales_count' => $daySales->count(),
                    'is_eligible' => false,
                    'note' => 'Monthly salary'
                ];
            }

            $currentDay->addDay();
        }

        return [
            'id' => $waiter->id,
            'name' => $waiter->name,
            'avatar' => $waiter->avatar,
            'weekly_total_sales' => $weeklySales->sum('total'),
            'weekly_total_payment' => 0, // No weekly payment
            'daily' => $dailyData,
            'payment_type' => 'monthly_salary',
            'special_note' => 'Paid fixed monthly salary (excluded from weekly payroll)',
            'excluded_from_weekly' => true
        ];
    }

    private function getDailyBreakdown($waiterId, $weekStart, $weekEnd)
    {
        $dailyData = [];
        $currentDay = $weekStart->copy();

        while ($currentDay <= $weekEnd) {
            $daySales = Sale::where('waiter_id', $waiterId)
                ->whereDate('created_at', $currentDay)
                ->with('items')
                ->get();

            $dayTotal = $daySales->sum('total');
            $dayPayment = $this->calculatePayment($dayTotal);

            // ✅ Only include days with sales
            if ($dayTotal > 0) {
                $dailyData[] = [
                    'date' => $currentDay->format('Y-m-d'),
                    'day_name' => $currentDay->format('l'),
                    'total_amount' => $dayTotal,
                    'payment' => $dayPayment,
                    'sales_count' => $daySales->count(),
                    'is_eligible' => $dayTotal >= 100000,
                    'note' => $dayTotal >= 100000 ? '4% commission' : 'Below threshold'
                ];
            }

            $currentDay->addDay();
        }

        return $dailyData;
    }

    private function calculatePayment($amount)
    {
        if ($amount < 100000) {
            return 0;
        }

        $fourPercent = $amount * 0.04;

        // Round to nearest 1000
        return round($fourPercent / 1000) * 1000;
    }

    private function getAvailableYears()
    {
        return Sale::selectRaw('YEAR(created_at) as year')
            ->groupBy('year')
            ->orderBy('year', 'desc')
            ->pluck('year')
            ->toArray();
    }

    public function export(Request $request)
    {
        if (auth()->user()->is_role != 1) {
            abort(403, 'Unauthorized access.');
        }

        $year = $request->get('year', Carbon::now()->year);
        $week = $request->get('week', Carbon::now()->weekOfYear);

        $weekData = $this->getWeekDates($year, $week);
        $weekStart = $weekData['start'];
        $weekEnd = $weekData['end'];

        // Get all waiters including special cases
        $waiters = User::whereIn('is_role', [2, 3])
            ->where('status', 1)
            ->get()
            ->map(function ($waiter) use ($weekStart, $weekEnd) {
                return $this->getWaiterPayrollData($waiter, $weekStart, $weekEnd);
            })
            ->filter(function ($waiter) {
                // Include waiters with sales OR Aisha Juma (even if no sales for reporting)
                return $waiter['weekly_total_sales'] > 0 || $waiter['name'] === 'aisha juma';
            })
            ->sortByDesc('weekly_total_sales')
            ->values();

            $filename =
                'Payroll_Report_Week_' . $week .
                '_(' . $weekStart->format('j_M') .
                '_to_' . $weekEnd->format('j_M') .
                '_' . $year . ').pdf';

            $pdf = PDF::loadView('admin.payroll.export', [
                'waiters' => $waiters,
                'week' => $week,
                'year' => $year,
                'weekStart' => $weekStart,
                'weekEnd' => $weekEnd,
                'exportTime' => now()->format('M j, Y g:i A')
            ]);

            return $pdf->download($filename);

    }

    public function getPaymentRanges()
    {
        $ranges = [];
        $amount = 100000;

        while ($amount <= 300000) {
            $ranges[] = [
                'base_amount' => $amount,
                'four_percent' => $amount * 0.04,
                'rounded_payment' => $this->calculatePayment($amount)
            ];
            $amount += 10000;
        }

        return $ranges;
    }
}