<?php
namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Customer;
use App\Models\Category;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\PosSetting;
use Illuminate\Http\Request;
use DB;
use Carbon\Carbon;

class PosController extends Controller
{
    // GET /api/product
    public function apiProducts(Request $request)
    {
        $q = $request->query('q');
        $query = Product::with('category')->orderBy('product_name');

        if ($q) {
            $query->where('product_name', 'like', "%{$q}%");
        }

        // send in a similar wrapper as the original vue expects
        $data = $query->paginate(100); // small pagination
        return response()->json(['data' => $data->items(), 'meta' => ['total' => $data->total()]]);
    }

    // GET /categorylist
    public function categoryList()
    {
        // Note: if your table is still named `category` singular change model/table accordingly.
        $categories = Category::orderBy('category_name')->get()->map(function($c) {
            return ['id' => $c->id, 'name' => $c->category_name];
        });
        return response()->json(['data' => $categories]);
    }

    // GET /category/product/{id}
    public function categoryProducts($id)
    {
        $products = Product::where('category_id', $id)->with('category')->get();
        return response()->json($products);
    }

    // GET /api/customers
    public function apiCustomers()
    {
        return response()->json(Customer::orderBy('name')->get());
    }

    // POST /sales
    public function createSale(Request $request)
    {
        $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'payment_type' => 'required|string',
            'grandtotal' => 'required|numeric|min:0',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $sale = Sale::create([
                'customer_id' => $request->customer_id,
                'user_id' => auth()->id() ?? null,
                'grandtotal' => $request->grandtotal,
                'total_payment' => $request->totalPayment ?? $request->grandtotal,
                'discount' => $request->discount ?? 0,
                'payment_type' => $request->payment_type,
                'comments' => $request->comments ?? null,
                'on_hold' => false,
            ]);

            foreach ($request->items as $it) {
                $sale->items()->create([
                    'product_id' => $it['product_id'],
                    'quantity' => $it['quantity'],
                    'price' => $it['price'],
                ]);

                // optional: decrease stock
                $product = Product::find($it['product_id']);
                if ($product && isset($product->stock)) {
                    $product->stock = max(0, $product->stock - $it['quantity']);
                    $product->save();
                }
            }

            DB::commit();
            return response()->json(['id' => $sale->id], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Could not create sale', 'error' => $e->getMessage()], 500);
        }
    }

    // POST /onholdsales
    public function onHoldSale(Request $request)
    {
        $request->merge(['on_hold' => true]);
        // reuse createSale pattern but set on_hold true
        DB::beginTransaction();
        try {
            $sale = Sale::create([
                'customer_id' => $request->customer_id,
                'user_id' => auth()->id() ?? null,
                'grandtotal' => $request->grandtotal,
                'total_payment' => $request->totalPayment ?? $request->grandtotal,
                'discount' => $request->discount ?? 0,
                'payment_type' => $request->payment_type,
                'comments' => $request->comments ?? null,
                'on_hold' => true,
            ]);

            foreach ($request->items as $it) {
                $sale->items()->create([
                    'product_id' => $it['product_id'],
                    'quantity' => $it['quantity'],
                    'price' => $it['price'],
                ]);
            }

            DB::commit();
            return response()->json(['id' => $sale->id], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Could not create on-hold sale', 'error' => $e->getMessage()], 500);
        }
    }

    // GET /sales/receipt/{id} (web)
    public function receipt($id)
    {
        $sale = Sale::with('items.product','customer')->findOrFail($id);
        $setting = PosSetting::first() ?: new PosSetting();
        $cashier = auth()->user() ? auth()->user()->name : 'Cashier';
        return view('sales.receipt', compact('sale','setting','cashier'));
    }

    // GET /sales/print_receipt/{id} (JSON used by modal)
    public function printReceiptJson($id)
    {
        $sale = Sale::with('items.product','customer')->findOrFail($id);
        $setting = PosSetting::first() ?: new PosSetting();
        $cashier = auth()->user() ? auth()->user()->name : 'Cashier';
        return response()->json([
            'sale' => $sale,
            'details' => $sale->items->map(function($it){
                return ['name' => $it->product->product_name ?? 'N/A', 'qty' => $it->quantity, 'price' => $it->price];
            }),
            'setting' => $setting,
            'cashier' => $cashier,
            'today' => now()->toDateString()
        ]);
    }
}
