<?php

namespace App\Http\Controllers;

use App\Models\Stocktaking;
use App\Models\Product;
use Illuminate\Http\Request;
use Carbon\Carbon;
use PDF;
use Excel;
use App\Exports\StocktakingExport;
use Illuminate\Support\Facades\DB;

class StocktakingReportController extends Controller
{
    public function index()
    {
        return view('stocktaking.report');
    }

    /**
     * Get report data for AJAX requests (on-screen table)
     */
    public function getReportData(Request $request)
    {
        try {
            $type = $request->get('type', 'day');
            $date = $request->get('date', Carbon::today()->format('Y-m-d'));

            [$startDate, $endDate, $title] = $this->getDateRangeAndTitle($type, $date);

            $stocktakings = $this->getStockData($startDate, $endDate);

            return response()->json([
                'success' => true,
                'data' => $stocktakings,
                'title' => $title,
                'type' => $type,
                'date' => $date
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading report data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export report to PDF/Excel/CSV
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'day');
        $date = $request->get('date', Carbon::today()->format('Y-m-d'));
        $format = $request->get('format', 'pdf');

        [$startDate, $endDate, $title] = $this->getDateRangeAndTitle($type, $date);

        $stocktakingData = $this->getStockData($startDate, $endDate);

        // Generate file name
        switch ($type) {
            case 'week':
                $week = $this->getWeekDates($date);
                $fileName = 'stocktaking-report-week-' . $week['start'] . '-to-' . $week['end'];
                break;
            case 'month':
                $fileName = 'stocktaking-report-' . Carbon::parse($date)->format('F-Y');
                break;
            default:
                $fileName = 'stocktaking-report-' . $date;
        }

        if ($format === 'pdf') {
            $pdf = PDF::loadView('stocktaking.export-pdf', compact('stocktakingData', 'title'));
            return $pdf->download($fileName . '.pdf');
        }

        return Excel::download(new StocktakingExport($type, $date), $fileName . '.' . $format);
    }

    /**
     * Helper to get date range and title for reports
     */
    private function getDateRangeAndTitle($type, $date)
    {
        switch ($type) {
            case 'week':
                $week = $this->getWeekDates($date);
                $startDate = Carbon::parse($week['start'])->startOfDay();
                $endDate = Carbon::parse($week['end'])->endOfDay();
                $title = "Stocktaking Report - Week of " . Carbon::parse($week['start'])->format('M d') . ' to ' . Carbon::parse($week['end'])->format('M d, Y');
                break;

            case 'month':
                $startDate = Carbon::parse($date)->startOfMonth()->startOfDay();
                $endDate = Carbon::parse($date)->endOfMonth()->endOfDay();
                $title = "Stocktaking Report - " . Carbon::parse($date)->format('F Y');
                break;

            default:
                $startDate = Carbon::parse($date)->startOfDay();
                $endDate = Carbon::parse($date)->endOfDay();
                $title = "Stocktaking Report - " . Carbon::parse($date)->format('F d, Y');
        }

        return [$startDate, $endDate, $title];
    }

    /**
     * Helper to calculate sold quantity for a product in a date range
     */
    private function getSoldQuantity($productId, $startDate, $endDate)
    {
        return DB::table('sale_items as si')
            ->join('sales as s', 'si.sale_id', '=', 's.id')
            ->where('si.product_id', $productId)
            ->whereBetween('s.created_at', [$startDate, $endDate])
            ->sum('si.quantity');
    }

    /**
     * Fetch stocktaking data with calculated fields
     */
    private function getStockData($startDate, $endDate)
    {
        $stocktakings = Stocktaking::with('product')
            ->join('products', 'stocktakings.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->where('categories.category_name', 'Breakfast')
            ->whereBetween('stocktakings.date', [$startDate, $endDate])
            ->where('stocktakings.opening_stock', '>', 0) // exclude zero opening stock
            ->select('stocktakings.*', 'products.product_name', 'products.purchase_price', 'products.selling_price')
            ->orderBy('products.product_name')
            ->get();

        return $stocktakings->map(function ($stock) use ($startDate, $endDate) {
            $soldQuantity = $this->getSoldQuantity($stock->product_id, $startDate, $endDate);
            $amountObtained = $soldQuantity * $stock->selling_price;
            $profit = $amountObtained - ($soldQuantity * $stock->buying_price);
            $closedStock = ($stock->opening_stock + $stock->additional_stock) - $soldQuantity;

            return [
                'id' => $stock->id,
                'product_name' => $stock->product_name,
                'date' => $stock->date->format('Y-m-d'),
                'opening_stock' => (int)$stock->opening_stock,
                'additional_stock' => (int)$stock->additional_stock,
                'buying_price' => (float)$stock->buying_price,
                'selling_price' => (float)$stock->selling_price,
                'sold_quantity' => (int)$soldQuantity,
                'amount_obtained' => (float)$amountObtained,
                'profit' => (float)$profit,
                'closed_stock' => (int)$closedStock
            ];
        });
    }

    /**
     * Helper to get week start and end dates
     */
    private function getWeekDates($date)
    {
        $carbonDate = Carbon::parse($date);
        return [
            'start' => $carbonDate->startOfWeek()->format('Y-m-d'),
            'end' => $carbonDate->endOfWeek()->format('Y-m-d')
        ];
    }
}
