<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Sale;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use PDF;
use DateTime;

class WaiterReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        // Only allow admin access
        if (auth()->user()->is_role != 1) {
            abort(403, 'Unauthorized access.');
        }

        $filter = $request->get('filter', 'today');
        $date = $request->get('date', Carbon::today()->format('Y-m-d'));
        $week = $request->get('week', Carbon::now()->weekOfYear);
        $month = $request->get('month', Carbon::now()->month);
        $year = $request->get('year', Carbon::now()->year);

        // Get all active waiters with current filter using proper DB queries
        $waiters = User::where('is_role', 3)
            ->where('status', 1)
            ->get()
            ->map(function($waiter) use ($filter, $date, $week, $month, $year) {
                return $this->getWaiterPerformanceData($waiter, $filter, $date, $week, $month, $year);
            })
            ->sortByDesc('total_sales_amount')
            ->values();

        // Get summary data for all time periods
        $summaryData = $this->getSummaryData();

        return view('admin.waiters-reports.index', compact(
            'waiters', 'filter', 'date', 'week', 'month', 'year', 'summaryData'
        ));
    }

    private function getSummaryData()
    {
        $today = Carbon::today();
        $weekStart = Carbon::now()->startOfWeek();
        $weekEnd = Carbon::now()->endOfWeek();
        $monthStart = Carbon::now()->startOfMonth();
        $monthEnd = Carbon::now()->endOfMonth();

        // Today's data using efficient queries
        $todayData = Sale::whereDate('created_at', $today)
            ->select([
                DB::raw('COUNT(*) as total_sales'),
                DB::raw('SUM(total) as total_amount'),
                DB::raw('COUNT(DISTINCT waiter_id) as active_waiters')
            ])
            ->first();

        $todayItems = Sale::whereDate('created_at', $today)
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });

        // This week's data
        $weekData = Sale::whereBetween('created_at', [$weekStart, $weekEnd])
            ->select([
                DB::raw('COUNT(*) as total_sales'),
                DB::raw('SUM(total) as total_amount')
            ])
            ->first();

        $weekItems = Sale::whereBetween('created_at', [$weekStart, $weekEnd])
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });

        // This month's data
        $monthData = Sale::whereBetween('created_at', [$monthStart, $monthEnd])
            ->select([
                DB::raw('COUNT(*) as total_sales'),
                DB::raw('SUM(total) as total_amount')
            ])
            ->first();

        $monthItems = Sale::whereBetween('created_at', [$monthStart, $monthEnd])
            ->with('items')
            ->get()
            ->sum(function($sale) {
                return $sale->items->sum('quantity');
            });

        // Get top performers
        $todayLeaders = $this->getTopPerformers($today);
        $weekLeaders = $this->getTopPerformers($weekStart, $weekEnd);
        $monthLeaders = $this->getTopPerformers($monthStart, $monthEnd);

        return [
            'today' => [
                'total_sales' => $todayData->total_sales ?? 0,
                'total_amount' => $todayData->total_amount ?? 0,
                'total_items' => $todayItems,
                'active_waiters' => $todayData->active_waiters ?? 0
            ],
            'week' => [
                'total_sales' => $weekData->total_sales ?? 0,
                'total_amount' => $weekData->total_amount ?? 0,
                'total_items' => $weekItems,
                'average_daily' => ($weekData->total_amount ?? 0) / 7
            ],
            'month' => [
                'total_sales' => $monthData->total_sales ?? 0,
                'total_amount' => $monthData->total_amount ?? 0,
                'total_items' => $monthItems,
                'average_daily' => ($monthData->total_amount ?? 0) / $monthStart->daysInMonth
            ],
            'today_leaders' => $todayLeaders,
            'week_leaders' => $weekLeaders,
            'month_leaders' => $monthLeaders,
        ];
    }

    private function getTopPerformers($startDate, $endDate = null)
    {
        $query = Sale::whereDate('created_at', '>=', $startDate);
        
        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        $performers = $query->with(['waiter', 'items'])
            ->get()
            ->groupBy('waiter_id')
            ->map(function($waiterSales, $waiterId) {
                $waiter = $waiterSales->first()->waiter;
                return [
                    'name' => $waiter ? $waiter->name : 'Unknown',
                    'sales' => $waiterSales->count(),
                    'amount' => $waiterSales->sum('total'),
                    'items' => $waiterSales->sum(function($sale) {
                        return $sale->items->sum('quantity');
                    })
                ];
            })
            ->sortByDesc('amount')
            ->values();

        return $performers;
    }

   public function show(Request $request, User $waiter)
{
    // Only allow admin access
    if (auth()->user()->is_role != 1) {
        abort(403, 'Unauthorized access.');
    }

    // Verify this is a waiter
    if ($waiter->is_role != 3 ) {
        abort(404, 'Waiter not found.');
    }

    $filter = $request->get('filter', 'today');
    $date = $request->get('date', Carbon::today()->format('Y-m-d'));
    $week = $request->get('week', Carbon::now()->weekOfYear);
    $month = $request->get('month', Carbon::now()->month);
    $year = $request->get('year', Carbon::now()->year);

    $performanceData = $this->getDetailedWaiterPerformance($waiter, $filter, $date, $week, $month, $year);
    
    // Get monthly performance data
    $monthlyPerformance = $this->getMonthlyPerformance($waiter->id, $year);
    $yearlyTotal = $this->getYearlyTotal($waiter->id, $year);

    // Get today's, weekly, and monthly stats for the waiter
    $todayStats = $this->getWaiterPeriodStats($waiter->id, 'today');
    $weeklyStats = $this->getWaiterPeriodStats($waiter->id, 'week');
    $monthlyStats = $this->getWaiterPeriodStats($waiter->id, 'month');

    // Paginate recent sales
    $sales = Sale::where('waiter_id', $waiter->id)
        ->with(['items.product', 'cashier'])
        ->orderBy('created_at', 'desc')
        ->paginate(15);

   $filter = $request->get('filter', 'today');
    $date = $request->get('date', Carbon::today()->format('Y-m-d'));
    $week = $request->get('week', Carbon::now()->weekOfYear);
    $month = $request->get('month', Carbon::now()->month);
    $year = $request->get('year', Carbon::now()->year);

    $performanceData = $this->getDetailedWaiterPerformance($waiter, $filter, $date, $week, $month, $year);
    
    // Get monthly performance data
    $monthlyPerformance = $this->getMonthlyPerformance($waiter->id, $year);
    $yearlyTotal = $this->getYearlyTotal($waiter->id, $year);

    // Get today's, weekly, and monthly stats for the waiter
    $todayStats = $this->getWaiterPeriodStats($waiter->id, 'today');
    $weeklyStats = $this->getWaiterPeriodStats($waiter->id, 'week');
    $monthlyStats = $this->getWaiterPeriodStats($waiter->id, 'month');

    // Get breakdown data for each period
    $todayBreakdown = $this->getPeriodBreakdownData($waiter->id, 'today');
    $weeklyBreakdown = $this->getPeriodBreakdownData($waiter->id, 'week');
    $monthlyBreakdown = $this->getPeriodBreakdownData($waiter->id, 'month');

    // Paginate recent sales
    $sales = Sale::where('waiter_id', $waiter->id)
        ->with(['items.product', 'cashier'])
        ->orderBy('created_at', 'desc')
        ->paginate(15);

    return view('admin.waiters-reports.show', array_merge($performanceData, [
        'waiter' => $waiter,
        'filter' => $filter,
        'date' => $date,
        'week' => $week,
        'month' => $month,
        'year' => $year,
        'monthlyPerformance' => $monthlyPerformance,
        'yearlyTotal' => $yearlyTotal,
        'todayStats' => $todayStats,
        'weeklyStats' => $weeklyStats,
        'monthlyStats' => $monthlyStats,
        'todayBreakdown' => $todayBreakdown,
        'weeklyBreakdown' => $weeklyBreakdown,
        'monthlyBreakdown' => $monthlyBreakdown,
        'sales' => $sales
    ]));
}
public function exportAll(Request $request)
{
    // Only allow admin access
    if (auth()->user()->is_role != 1) {
        abort(403, 'Unauthorized access.');
    }

    $filter = $request->get('filter', 'today');
    $date = $request->get('date', Carbon::today()->format('Y-m-d'));
    $week = $request->get('week', Carbon::now()->weekOfYear);
    $month = $request->get('month', Carbon::now()->month);
    $year = $request->get('year', Carbon::now()->year);

    // Get all waiters with performance data
    $waiters = User::where('is_role', 3)
        ->where('status', 1)
        ->get()
        ->map(function($waiter) use ($filter, $date, $week, $month, $year) {
            return $this->getWaiterPerformanceData($waiter, $filter, $date, $week, $month, $year);
        })
        ->sortByDesc('total_sales_amount')
        ->values();

    $filename = "all_waiters_performance_" . date('Y_m_d_H_i') . '.pdf';

    $pdf = PDF::loadView('admin.waiters-reports.export-all', [
        'waiters' => $waiters,
        'filter' => $filter,
        'date' => $date,
        'week' => $week,
        'month' => $month,
        'year' => $year,
        'summaryData' => $this->getSummaryData(),
        'exportTime' => now()->format('M j, Y g:i A')
    ]);

    return $pdf->download($filename);
}
private function getWaiterPeriodStats($waiterId, $period)
{
    $query = Sale::where('waiter_id', $waiterId);
    
    switch ($period) {
        case 'today':
            $query->whereDate('created_at', Carbon::today());
            $previousQuery = Sale::where('waiter_id', $waiterId)
                ->whereDate('created_at', Carbon::yesterday());
            break;
        case 'week':
            $startOfWeek = Carbon::now()->startOfWeek();
            $endOfWeek = Carbon::now()->endOfWeek();
            $query->whereBetween('created_at', [$startOfWeek, $endOfWeek]);
            
            $prevStartOfWeek = $startOfWeek->copy()->subWeek();
            $prevEndOfWeek = $prevStartOfWeek->copy()->endOfWeek();
            $previousQuery = Sale::where('waiter_id', $waiterId)
                ->whereBetween('created_at', [$prevStartOfWeek, $prevEndOfWeek]);
            break;
        case 'month':
            $startOfMonth = Carbon::now()->startOfMonth();
            $endOfMonth = Carbon::now()->endOfMonth();
            $query->whereBetween('created_at', [$startOfMonth, $endOfMonth]);
            
            $prevStartOfMonth = $startOfMonth->copy()->subMonth();
            $prevEndOfMonth = $prevStartOfMonth->copy()->endOfMonth();
            $previousQuery = Sale::where('waiter_id', $waiterId)
                ->whereBetween('created_at', [$prevStartOfMonth, $prevEndOfMonth]);
            break;
    }

    $currentSales = $query->with('items')->get();
    $previousSales = $previousQuery->with('items')->get();

    $currentStats = [
        'sales' => $currentSales->count(),
        'amount' => $currentSales->sum('total'),
        'items' => $currentSales->sum(function($sale) {
            return $sale->items->sum('quantity');
        })
    ];

    $previousStats = [
        'sales' => $previousSales->count(),
        'amount' => $previousSales->sum('total'),
        'items' => $previousSales->sum(function($sale) {
            return $sale->items->sum('quantity');
        })
    ];

    // Calculate growth percentages
    $currentStats['sales_growth'] = $this->calculateGrowth($previousStats['sales'], $currentStats['sales']);
    $currentStats['amount_growth'] = $this->calculateGrowth($previousStats['amount'], $currentStats['amount']);
    $currentStats['items_growth'] = $this->calculateGrowth($previousStats['items'], $currentStats['items']);

    // Calculate averages
    if ($period === 'week') {
        $currentStats['average_daily'] = $currentStats['amount'] / 7;
    } elseif ($period === 'month') {
        $currentStats['average_daily'] = $currentStats['amount'] / Carbon::now()->daysInMonth;
    } else {
        $currentStats['average_daily'] = $currentStats['amount'];
    }

    return $currentStats;
}
private function getPeriodBreakdownData($waiterId, $period)
{
    $query = Sale::where('waiter_id', $waiterId);
    
    switch ($period) {
        case 'today':
            $query->whereDate('created_at', Carbon::today());
            break;
        case 'week':
            $startOfWeek = Carbon::now()->startOfWeek();
            $endOfWeek = Carbon::now()->endOfWeek();
            $query->whereBetween('created_at', [$startOfWeek, $endOfWeek]);
            break;
        case 'month':
            $startOfMonth = Carbon::now()->startOfMonth();
            $endOfMonth = Carbon::now()->endOfMonth();
            $query->whereBetween('created_at', [$startOfMonth, $endOfMonth]);
            break;
    }

    $sales = $query->with(['items.product', 'cashier'])->get();
    
    // Basic stats
    $totalSales = $sales->count();
    $totalAmount = $sales->sum('total');
    $totalItems = $sales->sum(function($sale) {
        return $sale->items->sum('quantity');
    });

    // Payment method breakdown
    $paymentBreakdown = $sales->groupBy('payment_type')->map(function($group) {
        return [
            'count' => $group->count(),
            'amount' => $group->sum('total')
        ];
    });

    // Status breakdown
    $statusBreakdown = $sales->groupBy('status')->map(function($group) {
        return [
            'count' => $group->count(),
            'amount' => $group->sum('total')
        ];
    });

    // Top selling items
    $topItems = [];
    foreach ($sales as $sale) {
        foreach ($sale->items as $item) {
            $productName = $item->product->product_name ?? 'Unknown';
            if (!isset($topItems[$productName])) {
                $topItems[$productName] = [
                    'quantity' => 0,
                    'amount' => 0
                ];
            }
            $topItems[$productName]['quantity'] += $item->quantity;
            $topItems[$productName]['amount'] += $item->quantity * $item->price;
        }
    }
    arsort($topItems);

    return [
        'totalSales' => $totalSales,
        'totalAmount' => $totalAmount,
        'totalItems' => $totalItems,
        'paymentBreakdown' => $paymentBreakdown,
        'statusBreakdown' => $statusBreakdown,
        'topItems' => array_slice($topItems, 0, 10),
    ];
}
    private function getMonthlyPerformance($waiterId, $year)
    {
        $months = [];
        
        for ($month = 1; $month <= 12; $month++) {
            $monthData = [
                'month_name' => DateTime::createFromFormat('!m', $month)->format('F'),
                'weeks' => [],
                'total_sales' => 0,
                'total_amount' => 0
            ];

            // Define week ranges for the month
            $weekRanges = [
                [1, 7],    // Week 1: days 1-7
                [8, 14],   // Week 2: days 8-14  
                [15, 21],  // Week 3: days 15-21
                [22, 28],  // Week 4: days 22-28
                [29, 31]   // Week 5: days 29-31
            ];

            foreach ($weekRanges as $weekRange) {
                $startDay = $weekRange[0];
                $endDay = $weekRange[1];
                
                $startDate = Carbon::create($year, $month, $startDay)->startOfDay();
                $endDate = Carbon::create($year, $month, $endDay)->endOfDay();
                
                // Ensure we don't go beyond month end
                $monthEnd = Carbon::create($year, $month, 1)->endOfMonth();
                if ($endDate > $monthEnd) {
                    $endDate = $monthEnd;
                }
                
                // Skip if start date is beyond month end
                if ($startDate > $monthEnd) {
                    $weekData = ['sales' => 0, 'amount' => 0, 'items' => 0];
                } else {
                    $weekData = $this->getWeekData($waiterId, $startDate, $endDate);
                }
                
                $monthData['weeks'][] = $weekData;
                $monthData['total_sales'] += $weekData['sales'];
                $monthData['total_amount'] += $weekData['amount'];
            }

            $months[] = $monthData;
        }

        return $months;
    }

    private function getWeekData($waiterId, $startDate, $endDate)
    {
        $sales = Sale::where('waiter_id', $waiterId)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->with('items')
            ->get();

        return [
            'sales' => $sales->count(),
            'amount' => $sales->sum('total'),
            'items' => $sales->sum(function($sale) {
                return $sale->items->sum('quantity');
            })
        ];
    }

    private function getYearlyTotal($waiterId, $year)
    {
        $sales = Sale::where('waiter_id', $waiterId)
            ->whereYear('created_at', $year)
            ->with('items')
            ->get();

        return [
            'sales' => $sales->count(),
            'amount' => $sales->sum('total'),
            'items' => $sales->sum(function($sale) {
                return $sale->items->sum('quantity');
            })
        ];
    }

    private function applyDateFilter($query, $filter, $date, $week, $month, $year)
    {
        switch ($filter) {
            case 'today':
                $query->whereDate('created_at', $date);
                break;
            case 'yesterday':
                $query->whereDate('created_at', Carbon::parse($date)->subDay());
                break;
            case 'week':
                $startOfWeek = Carbon::now()->setISODate($year, $week)->startOfWeek();
                $endOfWeek = $startOfWeek->copy()->endOfWeek();
                $query->whereBetween('created_at', [$startOfWeek, $endOfWeek]);
                break;
            case 'month':
                $query->whereYear('created_at', $year)
                      ->whereMonth('created_at', $month);
                break;
            case 'year':
                $query->whereYear('created_at', $year);
                break;
            case 'custom':
                if ($date) {
                    $query->whereDate('created_at', $date);
                }
                break;
        }
    }

    private function getWaiterPerformanceData($waiter, $filter, $date, $week, $month, $year)
    {
        $baseQuery = Sale::where('waiter_id', $waiter->id);
        $this->applyDateFilter($baseQuery, $filter, $date, $week, $month, $year);

        $sales = $baseQuery->with('items')->get();
        
        $salesCount = $sales->count();
        $totalAmount = $sales->sum('total');
        $totalItems = $sales->sum(function($sale) {
            return $sale->items->sum('quantity');
        });

        // Calculate metrics
        $averageSale = $salesCount > 0 ? $totalAmount / $salesCount : 0;
        $performanceScore = ($totalAmount / 1000);

        // Get comparison data
        $comparisonData = $this->getComparisonData($waiter->id, $filter, $date, $week, $month, $year);

        return [
            'id' => $waiter->id,
            'name' => $waiter->name,
            'avatar' => $waiter->avatar,
            'total_sales_count' => $salesCount,
            'total_sales_amount' => $totalAmount,
            'total_items_sold' => $totalItems,
            'average_sale_value' => $averageSale,
            'performance_score' => $performanceScore,
            'sales_growth' => $comparisonData['sales_growth'],
            'amount_growth' => $comparisonData['amount_growth'],
            'items_growth' => $comparisonData['items_growth'],
        ];
    }

    private function getDetailedWaiterPerformance($waiter, $filter, $date, $week, $month, $year)
    {
        $baseQuery = Sale::where('waiter_id', $waiter->id);
        $this->applyDateFilter($baseQuery, $filter, $date, $week, $month, $year);

        $sales = $baseQuery->with(['items.product', 'cashier'])->get();
        
        // Basic stats
        $totalSales = $sales->count();
        $totalAmount = $sales->sum('total');
        $totalItems = $sales->sum(function($sale) {
            return $sale->items->sum('quantity');
        });
        $averageSale = $totalSales > 0 ? $totalAmount / $totalSales : 0;

        // Payment method breakdown
        $paymentBreakdown = $sales->groupBy('payment_type')->map(function($group) {
            return [
                'count' => $group->count(),
                'amount' => $group->sum('total')
            ];
        });

        // Status breakdown
        $statusBreakdown = $sales->groupBy('status')->map(function($group) {
            return [
                'count' => $group->count(),
                'amount' => $group->sum('total')
            ];
        });

        // Hourly performance (for today/yesterday/custom filters)
        $hourlyPerformance = [];
        if (in_array($filter, ['today', 'yesterday', 'custom'])) {
            for ($hour = 7; $hour <= 22; $hour++) {
                $hourSales = $sales->filter(function($sale) use ($hour) {
                    return $sale->created_at->hour == $hour;
                });
                
                $hourlyPerformance[$hour] = [
                    'sales' => $hourSales->count(),
                    'amount' => $hourSales->sum('total'),
                    'items' => $hourSales->sum(function($sale) {
                        return $sale->items->sum('quantity');
                    })
                ];
            }
        }

        // Top selling items
        $topItems = [];
        foreach ($sales as $sale) {
            foreach ($sale->items as $item) {
                $productName = $item->product->product_name ?? 'Unknown';
                if (!isset($topItems[$productName])) {
                    $topItems[$productName] = [
                        'quantity' => 0,
                        'amount' => 0
                    ];
                }
                $topItems[$productName]['quantity'] += $item->quantity;
                $topItems[$productName]['amount'] += $item->quantity * $item->price;
            }
        }
        arsort($topItems);

        // Performance metrics
        $comparisonData = $this->getComparisonData($waiter->id, $filter, $date, $week, $month, $year);
        
        $efficiencyScore = $this->calculateEfficiencyScore($sales);
        $customerValueScore = $this->calculateCustomerValueScore($sales);

        return [
            'totalSales' => $totalSales,
            'totalAmount' => $totalAmount,
            'totalItems' => $totalItems,
            'averageSale' => $averageSale,
            'paymentBreakdown' => $paymentBreakdown,
            'statusBreakdown' => $statusBreakdown,
            'hourlyPerformance' => $hourlyPerformance,
            'topItems' => array_slice($topItems, 0, 10),
            'salesGrowth' => $comparisonData['sales_growth'],
            'amountGrowth' => $comparisonData['amount_growth'],
            'itemsGrowth' => $comparisonData['items_growth'],
            'efficiencyScore' => $efficiencyScore,
            'customerValueScore' => $customerValueScore,
        ];
    }

    private function getComparisonData($waiterId, $filter, $date, $week, $month, $year)
    {
        $currentQuery = Sale::where('waiter_id', $waiterId);
        $previousQuery = Sale::where('waiter_id', $waiterId);

        switch ($filter) {
            case 'today':
                $currentQuery->whereDate('created_at', $date);
                $previousQuery->whereDate('created_at', Carbon::parse($date)->subDay());
                break;
            case 'yesterday':
                $currentQuery->whereDate('created_at', Carbon::parse($date)->subDay());
                $previousQuery->whereDate('created_at', Carbon::parse($date)->subDays(2));
                break;
            case 'week':
                $startOfWeek = Carbon::now()->setISODate($year, $week)->startOfWeek();
                $endOfWeek = $startOfWeek->copy()->endOfWeek();
                $currentQuery->whereBetween('created_at', [$startOfWeek, $endOfWeek]);
                
                $prevStartOfWeek = $startOfWeek->copy()->subWeek();
                $prevEndOfWeek = $prevStartOfWeek->copy()->endOfWeek();
                $previousQuery->whereBetween('created_at', [$prevStartOfWeek, $prevEndOfWeek]);
                break;
            case 'month':
                $currentQuery->whereYear('created_at', $year)->whereMonth('created_at', $month);
                $previousMonth = $month - 1;
                $previousYear = $year;
                if ($previousMonth == 0) {
                    $previousMonth = 12;
                    $previousYear = $year - 1;
                }
                $previousQuery->whereYear('created_at', $previousYear)->whereMonth('created_at', $previousMonth);
                break;
            case 'year':
                $currentQuery->whereYear('created_at', $year);
                $previousQuery->whereYear('created_at', $year - 1);
                break;
        }

        $currentSales = $currentQuery->with('items')->get();
        $previousSales = $previousQuery->with('items')->get();

        return [
            'sales_growth' => $this->calculateGrowth($previousSales->count(), $currentSales->count()),
            'amount_growth' => $this->calculateGrowth($previousSales->sum('total'), $currentSales->sum('total')),
            'items_growth' => $this->calculateGrowth(
                $previousSales->sum(function($s) { return $s->items->sum('quantity'); }),
                $currentSales->sum(function($s) { return $s->items->sum('quantity'); })
            ),
        ];
    }

    private function calculateGrowth($previous, $current)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        return round((($current - $previous) / $previous) * 100, 2);
    }

    private function calculateEfficiencyScore($sales)
    {
        if ($sales->isEmpty()) return 0;
        
        $totalSales = $sales->count();
        $totalItems = $sales->sum(function($sale) {
            return $sale->items->sum('quantity');
        });
        
        $itemsPerSale = $totalItems / $totalSales;
        return min(100, max(0, ($itemsPerSale / 3) * 100));
    }

    private function calculateCustomerValueScore($sales)
    {
        if ($sales->isEmpty()) return 0;
        
        $totalAmount = $sales->sum('total');
        $totalSales = $sales->count();
        $averageSale = $totalAmount / $totalSales;
        
        return min(100, max(0, (($averageSale - 5000) / 5000) * 100));
    }

    public function export(Request $request, User $waiter)
    {
        if (auth()->user()->is_role != 1) {
            abort(403, 'Unauthorized access.');
        }

        $filter = $request->get('filter', 'today');
        $date = $request->get('date', Carbon::today()->format('Y-m-d'));
        $week = $request->get('week', Carbon::now()->weekOfYear);
        $month = $request->get('month', Carbon::now()->month);
        $year = $request->get('year', Carbon::now()->year);

        $filename = "waiter_performance_{$waiter->name}_" . date('Y_m_d_H_i') . '.pdf';

        $performanceData = $this->getDetailedWaiterPerformance($waiter, $filter, $date, $week, $month, $year);

        $pdf = PDF::loadView('admin.waiters-reports.export', array_merge($performanceData, [
            'waiter' => $waiter,
            'filter' => $filter,
            'date' => $date,
            'week' => $week,
            'month' => $month,
            'year' => $year
        ]));

        return $pdf->download($filename);
    }
}