<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class ImportSqlDumpSeeder extends Seeder
{
    public function run(): void
    {  
        $tables = [
            'sale_items',
            'stocktakings', 
            'sales',
            'products',
            'categories',
            'users',
        ];

        // Disable foreign key checks
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');

        foreach ($tables as $table) {
            // Use truncate to reset auto-increment values
            DB::table($table)->truncate();
        }

        // Re-enable foreign key checks
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        // Increase memory limit for large SQL file
        ini_set('memory_limit', '-1');

        $sqlFilePath = database_path('restaurant.sql');

        if (file_exists($sqlFilePath)) {
            // Read the SQL file
            $sql = file_get_contents($sqlFilePath);
            
            // Remove AUTO_INCREMENT values from CREATE TABLE statements
            $sql = preg_replace('/AUTO_INCREMENT=\d+/', 'AUTO_INCREMENT=1', $sql);
            
            // Split the SQL into individual statements and execute in correct order
            $this->executeSqlStatements($sql);
            
            $this->command->info('SQL dump imported successfully.');
        } else {
            $this->command->error("SQL dump file not found at: $sqlFilePath");
        }
    }

    protected function executeSqlStatements(string $sql): void
    {
        // Split SQL by semicolons, but be careful with semicolons in strings
        $statements = $this->splitSqlStatements($sql);
        
        // Define the correct order for table data insertion
        $tableOrder = [
            'categories',
            'users', 
            'products',
            'sales',
            'sale_items',
            'stocktakings'
        ];
        
        $tableData = [];
        
        // Group INSERT statements by table
        foreach ($statements as $statement) {
            $statement = trim($statement);
            if (empty($statement)) continue;
            
            if (preg_match('/^INSERT INTO `([^`]+)`/i', $statement, $matches)) {
                $tableName = $matches[1];
                $tableData[$tableName][] = $statement;
            } else {
                // Execute non-INSERT statements immediately
                if (!empty($statement) && !preg_match('/^\/\*!/', $statement)) {
                    try {
                        DB::statement($statement);
                    } catch (\Exception $e) {
                        $this->command->warn("Failed to execute: " . substr($statement, 0, 100) . "... Error: " . $e->getMessage());
                    }
                }
            }
        }
        
        // Execute INSERT statements in correct order
        foreach ($tableOrder as $table) {
            if (isset($tableData[$table])) {
                foreach ($tableData[$table] as $insertStatement) {
                    try {
                        DB::statement($insertStatement);
                    } catch (\Exception $e) {
                        $this->command->warn("Failed to insert into $table: " . $e->getMessage());
                    }
                }
            }
        }
    }

    protected function splitSqlStatements(string $sql): array
    {
        // Remove comments
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
        $sql = preg_replace('/--.*?$/m', '', $sql);
        
        // Split by semicolons, but not within quotes
        $statements = [];
        $current = '';
        $inString = false;
        $stringChar = '';
        
        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];
            
            if (($char === "'" || $char === '"') && !$inString) {
                $inString = true;
                $stringChar = $char;
                $current .= $char;
            } elseif ($char === $stringChar && $inString) {
                // Check if it's escaped
                if ($i > 0 && $sql[$i-1] === '\\') {
                    $current .= $char;
                } else {
                    $inString = false;
                    $current .= $char;
                }
            } elseif ($char === ';' && !$inString) {
                $statements[] = trim($current);
                $current = '';
            } else {
                $current .= $char;
            }
        }
        
        // Add the last statement if any
        if (!empty(trim($current))) {
            $statements[] = trim($current);
        }
        
        return array_filter($statements);
    }
}