@extends('layouts.admin')

@section('title', 'Category List')

@section('content')
<div class="container-xxl flex-grow-1 container-p-y">
    <h4 class="fw-bold py-3 mb-4">
        <span class="text-muted fw-light">Admin /</span> Categories
    </h4>

    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Category List</h5>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
                <i class="fas fa-plus me-2"></i> Add Category
            </button>
        </div>

        <!-- Add Category Modal -->
        <div class="modal fade" id="addCategoryModal" tabindex="-1" aria-labelledby="addCategoryLabel" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <form id="addCategoryForm" class="needs-validation" novalidate>
                        @csrf
                        <div class="modal-header">
                            <h5 class="modal-title" id="addCategoryLabel">Add Category</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            <div class="mb-3">
                                <label for="categoryName" class="form-label">Category Name</label>
                                <input type="text" name="category_name" id="categoryName" class="form-control" required>
                                <div class="invalid-feedback">Please enter a category name.</div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="submit" class="btn btn-primary">Save</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Edit Category Modal -->
        <div class="modal fade" id="editCategoryModal" tabindex="-1" aria-labelledby="editCategoryLabel" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <form id="editCategoryForm" class="needs-validation" novalidate>
                        @csrf
                        <input type="hidden" name="category_id" id="editCategoryId">
                        <div class="modal-header">
                            <h5 class="modal-title" id="editCategoryLabel">Edit Category</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            <div class="mb-3">
                                <label for="editCategoryName" class="form-label">Category Name</label>
                                <input type="text" name="category_name" id="editCategoryName" class="form-control" required>
                                <div class="invalid-feedback">Please enter a category name.</div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="submit" class="btn btn-primary">Update</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Delete Confirmation Modal -->
        <div class="modal fade" id="deleteCategoryModal" tabindex="-1" aria-labelledby="deleteCategoryModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="deleteCategoryModalLabel">Confirm Delete</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <p class="text-body-tertiary lh-lg mb-0">
                            Are you sure you want to delete category <b id="deleteCategoryName"></b>?
                        </p>
                    </div>
                    <div class="modal-footer">
                        <button class="btn btn-danger" id="confirmDeleteBtn" type="button">Delete</button>
                        <button class="btn btn-secondary" type="button" data-bs-dismiss="modal">Cancel</button>
                    </div>
                </div>
            </div>
        </div>

        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped" id="categoryTable">
                    <thead>
                        <tr>
                            <th class="text-center">#</th>
                            <th>Category Name</th>
                            <th>Created At</th>
                            <th>Updated At</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        {{-- Populated by Ajax --}}
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Toast container -->
<div class="toast-container position-fixed top-0 end-0 p-3" style="z-index: 9999"></div>
@endsection

@push('styles')
<style>
    /* Phoenix-compatible modal fixes */
    .modal {
        z-index: 1055 !important;
    }
    
    .modal-backdrop {
        z-index: 1050 !important;
    }
    
    .modal-dialog {
        z-index: 1055 !important;
    }
    
    .modal-content {
        z-index: 1055 !important;
    }

    /* Mobile-specific adjustments */
    @media (max-width: 768px) {
        .modal-dialog {
            margin: 1rem !important;
            max-width: calc(100% - 2rem) !important;
        }
        
        .modal-content {
            max-height: 90vh !important;
            overflow-y: auto !important;
        }
        
        .btn-group-sm .btn {
            padding: 0.25rem 0.5rem !important;
            margin: 0.125rem !important;
        }
    }
</style>
@endpush

@push('scripts')
<script>
document.addEventListener("DOMContentLoaded", function() {
    // Initialize variables
    let deleteCategoryId = null;
    let deleteCategoryName = null;
    const tableBody = $('#categoryTable tbody');

    // Date formatting function
    function formatDate(dateString) {
        if (!dateString) return "—";
        const date = new Date(dateString);
        return isNaN(date) ? "—" : date.toLocaleString();
    }

    // Load categories function
    function loadCategories() {
        $.ajax({
            url: "{{ url('/category/data') }}",
            type: "GET",
            dataType: "json",
            success: function(data) {
                tableBody.empty();
                data.forEach((category, index) => {
                    tableBody.append(`
                        <tr>
                            <td class="text-center">${index + 1}</td>
                            <td>${category.category_name}</td>
                            <td>${formatDate(category.created_at)}</td>
                            <td>${formatDate(category.updated_at)}</td>
                            <td>
                                <div class="btn-group btn-group-sm" role="group">
                                    <button class="btn btn-warning edit-btn" 
                                            data-id="${category.id}" 
                                            data-name="${category.category_name}">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button class="btn btn-danger delete-btn" 
                                            data-id="${category.id}" 
                                            data-name="${category.category_name}">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    `);
                });

                // Initialize DataTable with Phoenix-compatible settings
                if ($.fn.DataTable.isDataTable('#categoryTable')) {
                    $('#categoryTable').DataTable().destroy();
                }
                
                $('#categoryTable').DataTable({
                    pageLength: 10,
                    order: [[0, 'asc']],
                    responsive: true,
                    autoWidth: false,
                    language: {
                        search: "_INPUT_",
                        searchPlaceholder: "Search categories..."
                    },
                    dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>rt<"row"<"col-sm-12 col-md-6"i><"col-sm-12 col-md-6"p>>'
                });
            },
            error: function(err) {
                console.error("Error fetching categories:", err);
                showToast('Error loading categories', 'error');
            }
        });
    }

    // Toast notification function (Phoenix-compatible)
    function showToast(message, type = 'success') {
        const bgClass = type === 'success' ? 'bg-success' : 'bg-danger';
        const icon = type === 'success' ? 'fa-check' : 'fa-exclamation-triangle';
        
        const toastHTML = `
            <div class="toast align-items-center text-white ${bgClass} border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="fas ${icon} me-2"></i>${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
            </div>
        `;
        
        const $toast = $(toastHTML);
        $('.toast-container').append($toast);
        
        // Use Phoenix's toast initialization
        const toastElement = $toast[0];
        const toast = new bootstrap.Toast(toastElement, {
            animation: true,
            autohide: true,
            delay: 5000
        });
        
        toast.show();
        
        // Remove toast after it's hidden
        toastElement.addEventListener('hidden.bs.toast', function() {
            $toast.remove();
        });
    }

    // Add Category Form Submission
    $('#addCategoryForm').on('submit', function(e) {
        e.preventDefault();
        
        // Phoenix form validation
        if (!this.checkValidity()) {
            e.stopPropagation();
            $(this).addClass('was-validated');
            return;
        }
        
        const formData = $(this).serialize();
        
        $.ajax({
            url: "{{ url('/category/store') }}",
            type: "POST",
            data: formData,
            success: function(response) {
                $('#addCategoryModal').modal('hide');
                $('#addCategoryForm')[0].reset();
                $('#addCategoryForm').removeClass('was-validated');
                loadCategories();
                showToast(`Category <b>${response.category_name}</b> added successfully!`, 'success');
            },
            error: function(xhr) {
                console.error('Error adding category:', xhr);
                showToast('Error saving category. Please try again.', 'error');
            }
        });
    });

    // Open Edit Modal
    $(document).on('click', '.edit-btn', function() {
        const id = $(this).data('id');
        const name = $(this).data('name');
        $('#editCategoryId').val(id);
        $('#editCategoryName').val(name);
        $('#editCategoryModal').modal('show');
    });

    // Edit Category Form Submission
    $('#editCategoryForm').on('submit', function(e) {
        e.preventDefault();
        
        // Phoenix form validation
        if (!this.checkValidity()) {
            e.stopPropagation();
            $(this).addClass('was-validated');
            return;
        }
        
        const id = $('#editCategoryId').val();
        const formData = $(this).serialize();
        
        $.ajax({
            url: "{{ url('/category/update') }}/" + id,
            type: "PUT",
            data: formData,
            success: function(response) {
                $('#editCategoryModal').modal('hide');
                $('#editCategoryForm')[0].reset();
                $('#editCategoryForm').removeClass('was-validated');
                loadCategories();
                showToast(`Category <b>${response.category_name}</b> updated successfully!`, 'success');
            },
            error: function(xhr) {
                console.error('Error updating category:', xhr);
                showToast('Error updating category. Please try again.', 'error');
            }
        });
    });

    // Open Delete Modal
    $(document).on('click', '.delete-btn', function() {
        deleteCategoryId = $(this).data('id');
        deleteCategoryName = $(this).data('name');
        $('#deleteCategoryName').text(deleteCategoryName);
        $('#deleteCategoryModal').modal('show');
    });

    // Confirm Delete
    $('#confirmDeleteBtn').on('click', function() {
        if (!deleteCategoryId) return;

        $.ajax({
            url: "{{ url('/category/delete') }}/" + deleteCategoryId,
            type: "DELETE",
            data: { 
                _token: "{{ csrf_token() }}" 
            },
            success: function(response) {
                $('#deleteCategoryModal').modal('hide');
                loadCategories();
                showToast(`Category <b>${deleteCategoryName}</b> deleted successfully!`, 'success');
                
                // Reset delete variables
                deleteCategoryId = null;
                deleteCategoryName = null;
            },
            error: function(xhr) {
                $('#deleteCategoryModal').modal('hide');
                console.error('Error deleting category:', xhr);
                showToast('Error deleting category. Please try again.', 'error');
            }
        });
    });

    // Phoenix modal compatibility
    function initializePhoenixModals() {
        // Let Phoenix handle modal initialization
        if (window.phoenix && window.phoenix.modalInit) {
            window.phoenix.modalInit();
        }
        
        // Ensure modals work with Phoenix theme
        $('.modal').on('show.bs.modal', function() {
            // Phoenix-compatible modal show handling
            $(this).css('display', 'block');
            setTimeout(() => {
                $(this).addClass('show');
            }, 150);
        });
    }

    // Initialize everything
    function initializePage() {
        loadCategories();
        initializePhoenixModals();
        
        // Initialize Phoenix components
        if (window.phoenix) {
            // Initialize form validation
            if (window.phoenix.formValidationInit) {
                window.phoenix.formValidationInit();
            }
            
            // Initialize tooltips
            if (window.phoenix.tooltipInit) {
                window.phoenix.tooltipInit();
            }
        }
    }

    // Start the page
    initializePage();
});
</script>
@endpush