@extends('layouts.admin')

@section('title', 'Cashier Stock Entry')

@section('content')
<div class="container-xxl flex-grow-1 container-p-y">
    <!-- Header -->
    <div class="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center mb-4">
        <div class="mb-3 mb-md-0">
            <h4 class="fw-bold mb-2">
                <span class="text-muted fw-light">Inventory /</span> Daily Stock Entry
            </h4>
            <p class="text-body-tertiary mb-0">Fill opening stock and additional stock for today</p>
        </div>
        <div class="d-flex flex-wrap gap-2">
            <div class="badge bg-primary rounded-pill fs-9 px-3 py-2">
                <i class="fas fa-calendar me-2"></i>
                {{ date('F d, Y') }}
            </div>
            <button class="btn btn-success btn-sm" id="saveAllBtn">
                <i class="fas fa-save me-2"></i>Save All
            </button>
        </div>
    </div>

    <!-- Info Alert -->
    <div class="alert alert-info mb-4">
        <i class="fas fa-info-circle me-2"></i>
        <strong>Instructions:</strong> 
        <ul class="mb-0 mt-2">
            <li>Fill <strong>Opening Stock</strong> only once at the beginning of the day (cannot be modified later)</li>
            <li>Update <strong>Additional Stock</strong> throughout the day as needed</li>
            <li>Other fields are automatically calculated from sales data</li>
        </ul>
    </div>

    <!-- Stock Entry Table -->
    <div class="card">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-striped" id="stockEntryTable">
                    <thead class="bg-body-secondary">
                        <tr>
                            <th width="5%">Sn</th>
                            <th width="20%">Product Name</th>
                            <th width="10%">Opening Stock</th>
                            <th width="10%">Additional Stock</th>
                            <th width="10%">Buying Price</th>
                            <th width="10%">Selling Price</th>
                            <th width="10%">Sold Today</th>
                            <th width="10%">Amount</th>
                            <th width="10%">Profit</th>
                            <th width="10%">Closed Stock</th>
                        </tr>
                    </thead>
                    <tbody id="stockEntryBody">
                        <!-- Data will be loaded here -->
                    </tbody>
                    <tfoot class="bg-body-secondary fw-bold">
                        <tr>
                            <td colspan="6" class="text-end">TOTALS:</td>
                            <td id="totalSold">0</td>
                            <td id="totalAmount">Tsh 0</td>
                            <td id="totalProfit">Tsh 0</td>
                            <td id="totalClosed">0</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Toast container -->
<div class="toast-container position-fixed top-0 end-0 p-3" style="z-index: 2000"></div>
@endsection

@push('scripts')
<script>
document.addEventListener("DOMContentLoaded", function() {
    const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
    const stockEntryBody = document.getElementById('stockEntryBody');
    const saveAllBtn = document.getElementById('saveAllBtn');

    let stockData = [];

    // Load stock data
    function loadStockData() {
        fetch('/cashier-stock/data', {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (Array.isArray(data)) {
                stockData = data;
                renderStockTable(data);
            } else {
                showToast('Error loading stock data', 'error');
            }
        })
        .catch(error => {
            console.error('Error loading stock data:', error);
            showToast('Error loading stock data', 'error');
        });
    }

    // Render stock table
    function renderStockTable(data) {
        stockEntryBody.innerHTML = '';
        
        let totalSold = 0;
        let totalAmount = 0;
        let totalProfit = 0;
        let totalClosed = 0;

        data.forEach((item, index) => {
            const row = document.createElement('tr');
            const openingDisabled = item.opening_stock_filled ? 'disabled' : '';
            const openingClass = item.opening_stock_filled ? 'bg-light' : '';
            
            row.innerHTML = `
                <td class="text-center">${index + 1}</td>
                <td>
                    <strong>${item.product_name}</strong>
                    <input type="hidden" name="product_id" value="${item.product_id}">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm opening-stock ${openingClass}" 
                           value="${item.opening_stock}" min="0" ${openingDisabled}
                           data-filled="${item.opening_stock_filled}">
                    ${item.opening_stock_filled ? '<small class="text-muted d-block">Already filled</small>' : ''}
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm additional-stock" 
                           value="${item.additional_stock}" min="0">
                </td>
                <td class="text-end">Tsh ${item.buying_price.toLocaleString()}</td>
                <td class="text-end">Tsh ${item.selling_price.toLocaleString()}</td>
                <td class="text-center">${item.sold_quantity}</td>
                <td class="text-end">Tsh ${item.amount_obtained.toLocaleString()}</td>
                <td class="text-end">Tsh ${item.profit.toLocaleString()}</td>
                <td class="text-center">${item.closed_stock}</td>
            `;
            stockEntryBody.appendChild(row);

            totalSold += parseInt(item.sold_quantity) || 0;
            totalAmount += parseFloat(item.amount_obtained) || 0;
            totalProfit += parseFloat(item.profit) || 0;
            totalClosed += parseInt(item.closed_stock) || 0;
        });

        document.getElementById('totalSold').textContent = totalSold;
        document.getElementById('totalAmount').textContent = `Tsh ${totalAmount.toLocaleString()}`;
        document.getElementById('totalProfit').textContent = `Tsh ${totalProfit.toLocaleString()}`;
        document.getElementById('totalClosed').textContent = totalClosed;
    }

    // Save all stock data
    saveAllBtn.addEventListener('click', function() {
        const updates = [];
        const rows = stockEntryBody.querySelectorAll('tr');

        rows.forEach(row => {
            const productId = row.querySelector('input[name="product_id"]').value;
            const openingStock = row.querySelector('.opening-stock').value;
            const additionalStock = row.querySelector('.additional-stock').value;
            const openingFilled = row.querySelector('.opening-stock').getAttribute('data-filled') === 'true';

            // Only include opening stock if it wasn't already filled
            if (!openingFilled && openingStock > 0) {
                updates.push({
                    product_id: productId,
                    opening_stock: parseInt(openingStock),
                    additional_stock: parseInt(additionalStock)
                });
            } else {
                updates.push({
                    product_id: productId,
                    additional_stock: parseInt(additionalStock)
                });
            }
        });

        // Send updates sequentially to avoid overwhelming the server
        let successCount = 0;
        let errorCount = 0;

        function updateNext(index) {
            if (index >= updates.length) {
                if (errorCount > 0) {
                    showToast(`Saved ${successCount} items, ${errorCount} failed`, 'warning');
                } else {
                    showToast('All stock data saved successfully!', 'success');
                    loadStockData(); // Reload to reflect changes
                }
                return;
            }

            const update = updates[index];
            
            fetch('/cashier-stock/update', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(update)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    successCount++;
                } else {
                    errorCount++;
                    console.error('Error updating product:', update.product_id, data.message);
                }
                updateNext(index + 1);
            })
            .catch(error => {
                errorCount++;
                console.error('Error updating product:', update.product_id, error);
                updateNext(index + 1);
            });
        }

        if (updates.length > 0) {
            updateNext(0);
        } else {
            showToast('No changes to save', 'info');
        }
    });

    // Load data on page load
    loadStockData();

    // Toast function
    function showToast(message, type = 'success') {
        const toastHTML = `
            <div class="toast show align-items-center text-bg-${type === 'success' ? 'success' : type === 'warning' ? 'warning' : 'danger'} border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body small">${message}</div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
            </div>
        `;
        
        const toastContainer = document.querySelector('.toast-container');
        const toastElement = document.createElement('div');
        toastElement.innerHTML = toastHTML;
        toastContainer.appendChild(toastElement);
        
        const toast = new bootstrap.Toast(toastElement.querySelector('.toast'));
        toast.show();
        
        toastElement.addEventListener('hidden.bs.toast', function() {
            toastElement.remove();
        });
    }
});
</script>

<style>
.opening-stock:disabled {
    background-color: #e9ecef;
    opacity: 1;
}

.additional-stock {
    background-color: #fff3cd;
}

.table td {
    vertical-align: middle;
}

.bg-light {
    background-color: #f8f9fa !important;
}
</style>
@endpush