@extends('layouts.admin')

@section('title', 'Stocktaking')

@section('content')
<div class="container-xxl flex-grow-1 container-p-y">
    <!-- Header -->
    <div class="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center mb-4">
        <div class="mb-3 mb-md-0">
            <h4 class="fw-bold mb-2">
                <span class="text-muted fw-light">Inventory /</span> Stocktaking Form
            </h4>
            <p class="text-body-tertiary mb-0">Daily stock management and tracking</p>
        </div>
        <div class="d-flex flex-wrap gap-2">
            <div class="input-group input-group-sm" style="width: 200px;">
                <input type="date" class="form-control" id="stocktakingDate" value="{{ date('Y-m-d') }}">
                <button class="btn btn-outline-primary" type="button" id="loadStocktakingData">
                    <i class="fas fa-sync-alt"></i>
                </button>
            </div>
            <button class="btn btn-success btn-sm" id="saveStocktakingBtn">
                <i class="fas fa-save me-2"></i>Save Data
            </button>
            <button class="btn btn-warning btn-sm" id="resetOpeningStockBtn">
                <i class="fas fa-redo me-2"></i>Reset Opening Stock
            </button>
        </div>
    </div>

    <!-- Stocktaking Form -->
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">Acadogar Investments Limited - "D-Restaurant STOCKTAKING FORM"</h5>
            <p class="text-body-tertiary mb-0">
                Basement floor PSSSF Parking Building, Makole Dodoma | 
                Phone: +255 716 780 595, +255 626 374 520, +255 755 839108 | 
                Email: acadogar@gmail.com
            </p>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered table-striped" id="stocktakingTable">
                    <thead class="bg-body-secondary">
                        <tr>
                            <th width="5%">Sn</th>
                            <th width="20%">Description</th>
                            <th width="10%">Opening Stock</th>
                            <th width="10%">Additional</th>
                            <th width="10%">Buying Price</th>
                            <th width="10%">Selling Price</th>
                            <th width="10%">Sold</th>
                            <th width="10%">Amount</th>
                            <th width="10%">Profit</th>
                            <th width="10%">Closed Stock</th>
                        </tr>
                    </thead>
                    <tbody id="stocktakingBody">
                        <!-- Data will be loaded here -->
                    </tbody>
                    <tfoot class="bg-body-secondary fw-bold">
                        <tr>
                            <td colspan="6" class="text-end">TOTALS:</td>
                            <td id="totalSold">0</td>
                            <td id="totalAmount">Tsh 0</td>
                            <td id="totalProfit">Tsh 0</td>
                            <td id="totalClosed">0</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Toast container -->
<div class="toast-container position-fixed top-0 end-0 p-3" style="z-index: 2000"></div>
@endsection

@push('scripts')
<script>
document.addEventListener("DOMContentLoaded", function() {
    const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
    const stocktakingBody = document.getElementById('stocktakingBody');
    const stocktakingDate = document.getElementById('stocktakingDate');
    const loadStocktakingData = document.getElementById('loadStocktakingData');
    const saveStocktakingBtn = document.getElementById('saveStocktakingBtn');
    const resetOpeningStockBtn = document.getElementById('resetOpeningStockBtn');

    // Load stocktaking data
    function loadStocktakingDataForDate() {
        const date = stocktakingDate.value;
        
        fetch(`/stocktaking/data?date=${date}`, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            }
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            if (Array.isArray(data)) {
                renderStocktakingTable(data);
            } else {
                console.error('Expected array but got:', data);
                showToast('Invalid data format received', 'error');
            }
        })
        .catch(error => {
            console.error('Error loading stocktaking data:', error);
            showToast('Error loading stocktaking data', 'error');
        });
    }

    // Render stocktaking table
    function renderStocktakingTable(data) {
        if (!Array.isArray(data)) {
            console.error('Data is not an array:', data);
            return;
        }

        stocktakingBody.innerHTML = '';
        let totalSold = 0;
        let totalAmount = 0;
        let totalProfit = 0;
        let totalClosed = 0;

        data.forEach((item, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td class="text-center">${index + 1}</td>
                <td>
                    <strong>${item.product_name}</strong>
                    <input type="hidden" name="product_id" value="${item.product_id}">
                    <input type="hidden" name="id" value="${item.id}">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm opening-stock" 
                           value="${item.opening_stock}" min="0">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm additional-stock" 
                           value="${item.additional_stock}" min="0">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm buying-price" 
                           value="${item.buying_price}" step="0.01" min="0">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm selling-price" 
                           value="${item.selling_price}" step="0.01" min="0">
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm sold-quantity" 
                           value="${item.sold_quantity}" min="0">
                </td>
                <td class="text-end amount-obtained">Tsh ${item.amount_obtained.toLocaleString()}</td>
                <td class="text-end profit">Tsh ${item.profit.toLocaleString()}</td>
                <td class="text-end closed-stock">${item.closed_stock}</td>
            `;
            stocktakingBody.appendChild(row);

            // Update totals
            totalSold += parseInt(item.sold_quantity) || 0;
            totalAmount += parseFloat(item.amount_obtained) || 0;
            totalProfit += parseFloat(item.profit) || 0;
            totalClosed += parseInt(item.closed_stock) || 0;
        });

        // Update footer totals
        document.getElementById('totalSold').textContent = totalSold;
        document.getElementById('totalAmount').textContent = `Tsh ${totalAmount.toLocaleString()}`;
        document.getElementById('totalProfit').textContent = `Tsh ${totalProfit.toLocaleString()}`;
        document.getElementById('totalClosed').textContent = totalClosed;

        // Add event listeners for real-time calculations
        addCalculationEventListeners();
    }

    // Add event listeners for real-time calculations
    function addCalculationEventListeners() {
        const inputs = stocktakingBody.querySelectorAll('input[type="number"]');
        inputs.forEach(input => {
            input.addEventListener('input', function() {
                calculateRowTotals(this);
            });
        });
    }

    // Calculate row totals
    function calculateRowTotals(input) {
        const row = input.closest('tr');
        
        const openingStock = parseFloat(row.querySelector('.opening-stock').value) || 0;
        const additionalStock = parseFloat(row.querySelector('.additional-stock').value) || 0;
        const buyingPrice = parseFloat(row.querySelector('.buying-price').value) || 0;
        const sellingPrice = parseFloat(row.querySelector('.selling-price').value) || 0;
        const soldQuantity = parseFloat(row.querySelector('.sold-quantity').value) || 0;

        // Calculate derived values
        const amountObtained = soldQuantity * sellingPrice;
        const cost = soldQuantity * buyingPrice;
        const profit = amountObtained - cost;
        const closedStock = (openingStock + additionalStock) - soldQuantity;

        // Update display
        row.querySelector('.amount-obtained').textContent = `Tsh ${amountObtained.toLocaleString()}`;
        row.querySelector('.profit').textContent = `Tsh ${profit.toLocaleString()}`;
        row.querySelector('.closed-stock').textContent = Math.max(0, closedStock);

        // Update grand totals
        updateGrandTotals();
    }

    // Update grand totals
    function updateGrandTotals() {
        let totalSold = 0;
        let totalAmount = 0;
        let totalProfit = 0;
        let totalClosed = 0;

        const rows = stocktakingBody.querySelectorAll('tr');
        rows.forEach(row => {
            const soldQuantity = parseFloat(row.querySelector('.sold-quantity').value) || 0;
            const amountObtained = parseFloat(row.querySelector('.amount-obtained').textContent.replace('Tsh ', '').replace(/,/g, '')) || 0;
            const profit = parseFloat(row.querySelector('.profit').textContent.replace('Tsh ', '').replace(/,/g, '')) || 0;
            const closedStock = parseInt(row.querySelector('.closed-stock').textContent) || 0;

            totalSold += soldQuantity;
            totalAmount += amountObtained;
            totalProfit += profit;
            totalClosed += closedStock;
        });

        document.getElementById('totalSold').textContent = totalSold;
        document.getElementById('totalAmount').textContent = `Tsh ${totalAmount.toLocaleString()}`;
        document.getElementById('totalProfit').textContent = `Tsh ${totalProfit.toLocaleString()}`;
        document.getElementById('totalClosed').textContent = totalClosed;
    }

    // Save stocktaking data
    saveStocktakingBtn.addEventListener('click', function() {
        const stocktakingData = [];
        const rows = stocktakingBody.querySelectorAll('tr');

        rows.forEach(row => {
            const productId = row.querySelector('input[name="product_id"]').value;
            const recordId = row.querySelector('input[name="id"]').value;
            
            stocktakingData.push({
                product_id: productId,
                id: recordId,
                opening_stock: parseFloat(row.querySelector('.opening-stock').value) || 0,
                additional_stock: parseFloat(row.querySelector('.additional-stock').value) || 0,
                buying_price: parseFloat(row.querySelector('.buying-price').value) || 0,
                selling_price: parseFloat(row.querySelector('.selling-price').value) || 0,
                sold_quantity: parseFloat(row.querySelector('.sold-quantity').value) || 0
            });
        });

        fetch('/stocktaking/save', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify({
                stocktaking_data: stocktakingData,
                date: stocktakingDate.value
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast(data.message, 'success');
            } else {
                showToast(data.message, 'error');
            }
        })
        .catch(error => {
            console.error('Error saving stocktaking data:', error);
            showToast('Error saving stocktaking data', 'error');
        });
    });

    // Reset opening stock
    resetOpeningStockBtn.addEventListener('click', function() {
        if (confirm('Are you sure you want to reset opening stock for today?')) {
            fetch('/stocktaking/reset-opening', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showToast(data.message, 'success');
                    loadStocktakingDataForDate();
                } else {
                    showToast(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error resetting opening stock:', error);
                showToast('Error resetting opening stock', 'error');
            });
        }
    });

    // Event listeners
    loadStocktakingData.addEventListener('click', loadStocktakingDataForDate);
    
    // Load data on page load
    loadStocktakingDataForDate();

    // Toast function
    function showToast(message, type = 'success') {
        const toastHTML = `
            <div class="toast show align-items-center text-bg-${type === 'success' ? 'success' : 'danger'} border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body small">
                        <i class="fas fa-${type === 'success' ? 'check' : 'exclamation-triangle'} me-2"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                </div>
            </div>
        `;
        
        const toastContainer = document.querySelector('.toast-container');
        const toastElement = document.createElement('div');
        toastElement.innerHTML = toastHTML;
        toastContainer.appendChild(toastElement);
        
        const toast = new bootstrap.Toast(toastElement.querySelector('.toast'));
        toast.show();
        
        toastElement.addEventListener('hidden.bs.toast', function() {
            toastElement.remove();
        });
    }
});
</script>

<style>
.stocktaking-table input {
    border: 1px solid #dee2e6;
    text-align: center;
}

.stocktaking-table input:focus {
    border-color: #86b7fe;
    box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
}

.stocktaking-table td {
    vertical-align: middle;
}

.amount-obtained, .profit, .closed-stock {
    font-weight: 600;
    background-color: #f8f9fa;
}
</style>
@endpush