<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\SaleController;
use App\Http\Controllers\SaleReportController;
use App\Http\Controllers\StocktakingReportController;
use App\Http\Controllers\CashierStockController;
use App\Http\Controllers\WaiterReportController;
use App\Http\Controllers\PayrollController;
use App\Http\Controllers\ValidatorDashboardController; // ADDED

/*
|--------------------------------------------------------------------------
| Public Routes
|--------------------------------------------------------------------------
*/
Route::get('/', [AuthController::class, 'login'])->name('login');
Route::post('login_post', [AuthController::class, 'login_post']);
Route::get('logout', [AuthController::class, 'logout'])->name('logout');

/*
|--------------------------------------------------------------------------
| Dashboard (shared, content depends on role)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'dashboard'])->name('dashboard');
    Route::get('/validator/dashboard', [ValidatorDashboardController::class, 'dashboard'])->name('validator.dashboard');
    Route::get('/api/chart-data', [DashboardController::class, 'getChartData'])->name('chart.data');
});

Route::get('/pdf-test', function() {
    $pdf = PDF::loadHTML('<h1>Hello PDF</h1>');
    return $pdf->download('test.pdf');
});

/*
|--------------------------------------------------------------------------
| Sales Operations (authenticated users)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    // Sales main page and search
    Route::get('/sales', [SaleController::class, 'index'])->name('sales.index');
    Route::get('/sales/search', [SaleController::class, 'search'])->name('sales.search');
    
    // Sale verification
    Route::post('/sales/{sale}/verify', [DashboardController::class, 'verify'])->name('sales.verify');

    // Cart actions
    Route::post('/cart/add', [SaleController::class, 'addToCart'])->name('cart.add');
    Route::put('/cart/update', [SaleController::class, 'updateCart'])->name('cart.update');
    Route::delete('/cart/remove', [SaleController::class, 'removeFromCart'])->name('cart.remove');
    Route::post('/cart/clear', [SaleController::class, 'clearCart'])->name('cart.clear');

    // Receipt
    Route::get('/sales/receipt/{id}', [SaleController::class, 'showReceipt'])->name('sales.receipt');

    // Create sale
    Route::post('/sales', [SaleController::class, 'store'])->name('sales.store');

    // Filter products by category
    Route::get('/category/product/{id}', [SaleController::class, 'categoryProducts'])->name('category.products');

    // Chart data
    Route::get('/api/sales-chart-data', [SaleController::class, 'getChartData'])->name('sales.chart.data');
});

/*
|--------------------------------------------------------------------------
| Validator Routes (authenticated users with role 6)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    Route::prefix('validator')->group(function () {
        Route::get('/dashboard', [ValidatorDashboardController::class, 'dashboard'])->name('validator.dashboard');
        Route::post('/mark-attended/{sale}', [ValidatorDashboardController::class, 'markAsAttended'])->name('validator.mark-attended');
        Route::get('/validator/dashboard-data', [ValidatorDashboardController::class, 'getDashboardData'])->name('validator.dashboard.data');
        Route::get('/unattended-count', [ValidatorDashboardController::class, 'getUnattendedCount'])->name('validator.unattended-count');
    });
});


/*
|--------------------------------------------------------------------------
| Enhanced Sales Analytics & Reports (authenticated users)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    // Main report page
    Route::get('/sales/report', [SaleReportController::class, 'report'])->name('sales.report');
    
    // Data endpoints for different report types
    Route::get('/sales/report/data/detailed', [SaleReportController::class, 'reportData'])->name('sales.report.data.detailed');
    Route::get('/sales/report/data/daily', [SaleReportController::class, 'dailySalesData'])->name('sales.report.data.daily');
    Route::get('/sales/report/data/weekly', [SaleReportController::class, 'weeklySalesData'])->name('sales.report.data.weekly');
    Route::get('/sales/report/data/monthly', [SaleReportController::class, 'monthlySalesData'])->name('sales.report.data.monthly');
    Route::get('/sales/report/data/products', [SaleReportController::class, 'productPerformanceData'])->name('sales.report.data.products');
    Route::get('/sales/report/data/waiters', [SaleReportController::class, 'waiterPerformanceData'])->name('sales.report.data.waiters');
    Route::get('/sales/report/data/hourly', [SaleReportController::class, 'hourlySalesData'])->name('sales.report.data.hourly');
    Route::get('/sales/report/summary', [SaleReportController::class, 'getSummaryStats'])->name('sales.report.summary');
    
    // Export routes
    Route::get('/sales/report/export/excel', [SaleReportController::class, 'exportExcel'])->name('sales.report.export.excel');
    Route::get('/sales/report/export/pdf', [SaleReportController::class, 'exportPDF'])->name('sales.report.export.pdf');
    Route::get('/sales/report/export/csv', [SaleReportController::class, 'exportCSV'])->name('sales.report.export.csv');
    Route::get('/sales/report/export/csv-enhanced', [SaleReportController::class, 'exportCSVEnhanced'])->name('sales.report.export.csv-enhanced');
    Route::get('/sales/report/export/{reportType}', [SaleReportController::class, 'exportReportType'])->name('sales.report.export.type');
});

/*
|--------------------------------------------------------------------------
| Payroll Routes
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->prefix('admin')->group(function() {
    Route::get('payroll', [PayrollController::class, 'index'])->name('payroll.index');
    Route::get('payroll/export', [PayrollController::class, 'export'])->name('payroll.export');
});

/*
|--------------------------------------------------------------------------
| Stocktaking Report Routes
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    Route::get('/stocktaking-report', [StocktakingReportController::class, 'index'])->name('stocktaking.report');
    Route::get('/stocktaking-report/data', [StocktakingReportController::class, 'getReportData'])->name('stocktaking.report.data');
    Route::get('/stocktaking-report/export', [StocktakingReportController::class, 'export'])->name('stocktaking.report.export');
});

/*
|--------------------------------------------------------------------------
| Cashier Stock Entry Routes
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    Route::get('/cashier-stock', [CashierStockController::class, 'index'])->name('cashier.stock');
    Route::get('/cashier-stock/data', [CashierStockController::class, 'getTodayStockData'])->name('cashier.stock.data');
    Route::post('/cashier-stock/update', [CashierStockController::class, 'updateStock'])->name('cashier.stock.update');
    Route::post('/cashier-stock/reset', [CashierStockController::class, 'resetForNewDay'])->name('cashier.stock.reset');
});

/*
|--------------------------------------------------------------------------
| Waiters Reports Routes (Admin Only)
|--------------------------------------------------------------------------
*/
Route::middleware(['auth'])->group(function () {
    Route::get('/admin/waiters-reports', [WaiterReportController::class, 'index'])->name('admin.waiters.reports');
    Route::get('/admin/waiters-reports/{waiter}', [WaiterReportController::class, 'show'])->name('admin.waiters.reports.show');
    Route::get('/admin/waiters-reports/{waiter}/export', [WaiterReportController::class, 'export'])->name('admin.waiters.reports.export');
    Route::get('/admin/waiters-reports/export-all', [WaiterReportController::class, 'exportAll'])->name('admin.waiters.reports.export.all');
});

/*
|--------------------------------------------------------------------------
| Utility Routes
|--------------------------------------------------------------------------
*/
Route::get('/refresh-csrf', function() {
    return response()->json(['token' => csrf_token()]);
});

Route::get('/debug-sales-route', function () {
    return response()->json([
        'message' => 'Route is accessible',
        'url' => url('sales/report/data'),
        'full_url' => url('sales/report/data') . '?test=1'
    ]);
});

/*
|--------------------------------------------------------------------------
| Admin-only Routes
|--------------------------------------------------------------------------
*/
Route::middleware(['admin'])->group(function () {
    /*
    |--------------------------------------------------------------------------
    | Category Management
    |--------------------------------------------------------------------------
    */
    Route::get('/category', [CategoryController::class, 'index']);
    Route::get('/category/data', [CategoryController::class, 'getCategories']);
    Route::post('/category/store', [CategoryController::class, 'store']);
    Route::put('/category/update/{id}', [CategoryController::class, 'update']);
    Route::delete('/category/delete/{id}', [CategoryController::class, 'destroy']);

    /*
    |--------------------------------------------------------------------------
    | Product Management
    |--------------------------------------------------------------------------
    */
    Route::get('/product', [ProductController::class, 'index']);
    Route::get('/product/data', [ProductController::class, 'data']);
    Route::post('/product/store', [ProductController::class, 'store']);
    Route::put('/product/update/{id}', [ProductController::class, 'update']);
    Route::delete('/product/delete/{id}', [ProductController::class, 'destroy']);

    /*
    |--------------------------------------------------------------------------
    | User Management
    |--------------------------------------------------------------------------
    */
    Route::get('/users', [UserController::class, 'index'])->name('users.index');
    Route::post('/users', [UserController::class, 'store'])->name('users.store');
    Route::get('/users/edit/{id}', [UserController::class, 'edit']);
    Route::put('/users/update/{id}', [UserController::class, 'update']);
    Route::delete('/users/delete/{id}', [UserController::class, 'destroy']);

    /*
    |--------------------------------------------------------------------------
    | Sales Admin Actions
    |--------------------------------------------------------------------------
    */
    Route::delete('/sales/{sale}', [SaleController::class, 'destroy'])->name('sales.destroy');
});